Package["core-runtime"].queue("constraint-solver",function () {/* Imports */
var Meteor = Package.meteor.Meteor;
var global = Package.meteor.global;
var meteorEnv = Package.meteor.meteorEnv;
var EmitterPromise = Package.meteor.EmitterPromise;
var ECMAScript = Package.ecmascript.ECMAScript;
var check = Package.check.check;
var Match = Package.check.Match;
var PackageVersion = Package['package-version-parser'].PackageVersion;
var Logic = Package['logic-solver'].Logic;
var meteorInstall = Package.modules.meteorInstall;
var Promise = Package.promise.Promise;

/* Package-scope variables */
var ConstraintSolver, validatePackageName;

var require = meteorInstall({"node_modules":{"meteor":{"constraint-solver":{"datatypes.js":function module(){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// packages/constraint-solver/datatypes.js                                                                             //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
ConstraintSolver = {};
var PV = PackageVersion;
var CS = ConstraintSolver;

////////// PackageAndVersion

// An ordered pair of (package, version).
CS.PackageAndVersion = function (pkg, version) {
  check(pkg, String);
  check(version, String);
  this.package = pkg;
  this.version = version;
};

// The string form of a PackageAndVersion is "package version",
// for example "foo 1.0.1".  The reason we don't use an "@" is
// it would look too much like a PackageConstraint.
CS.PackageAndVersion.prototype.toString = function () {
  return this.package + " " + this.version;
};
CS.PackageAndVersion.fromString = function (str) {
  var parts = str.split(' ');
  if (parts.length === 2 && parts[0] && parts[1]) {
    return new CS.PackageAndVersion(parts[0], parts[1]);
  } else {
    throw new Error("Malformed PackageAndVersion: " + str);
  }
};

////////// Dependency

// A Dependency consists of a PackageConstraint (like "foo@=1.2.3")
// and flags, like "isWeak".

CS.Dependency = function (packageConstraint, flags) {
  if (typeof packageConstraint !== 'string') {
    // this `if` is because Match.OneOf is really, really slow when it fails
    check(packageConstraint, Match.OneOf(PV.PackageConstraint, String));
  }
  if (typeof packageConstraint === 'string') {
    packageConstraint = PV.parsePackageConstraint(packageConstraint);
  }
  if (flags) {
    check(flags, Object);
  }
  this.packageConstraint = packageConstraint;
  this.isWeak = false;
  if (flags) {
    if (flags.isWeak) {
      this.isWeak = true;
    }
  }
};

// The string form of a Dependency is `?foo@1.0.0` for a weak
// reference to package "foo" with VersionConstraint "1.0.0".
CS.Dependency.prototype.toString = function () {
  var ret = this.packageConstraint.toString();
  if (this.isWeak) {
    ret = '?' + ret;
  }
  return ret;
};
CS.Dependency.fromString = function (str) {
  var isWeak = false;
  if (str.charAt(0) === '?') {
    isWeak = true;
    str = str.slice(1);
  }
  var flags = isWeak ? {
    isWeak: true
  } : null;
  return new CS.Dependency(str, flags);
};
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"catalog-cache.js":function module(require){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// packages/constraint-solver/catalog-cache.js                                                                         //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
const has = require('lodash.has');
const memoize = require('lodash.memoize');
var CS = ConstraintSolver;
var PV = PackageVersion;
var pvkey = function (pkg, version) {
  return pkg + " " + version;
};

// Stores the Dependencies for each known PackageAndVersion.
CS.CatalogCache = function () {
  // String(PackageAndVersion) -> String -> Dependency.
  // For example, "foo 1.0.0" -> "bar" -> Dependency.fromString("?bar@1.0.2").
  this._dependencies = {};
  // A map derived from the keys of _dependencies, for ease of iteration.
  // "foo" -> ["1.0.0", ...]
  // Versions in the array are unique but not sorted, unless the `.sorted`
  // property is set on the array.  The array is never empty.
  this._versions = {};
};
CS.CatalogCache.prototype.hasPackageVersion = function (pkg, version) {
  return has(this._dependencies, pvkey(pkg, version));
};
CS.CatalogCache.prototype.addPackageVersion = function (p, v, deps) {
  check(p, String);
  check(v, String);
  // `deps` must not have any duplicate values of `.packageConstraint.package`
  check(deps, [CS.Dependency]);
  var key = pvkey(p, v);
  if (has(this._dependencies, key)) {
    throw new Error("Already have an entry for " + key);
  }
  if (!has(this._versions, p)) {
    this._versions[p] = [];
  }
  this._versions[p].push(v);
  this._versions[p].sorted = false;
  var depsByPackage = {};
  this._dependencies[key] = depsByPackage;
  deps.forEach(function (d) {
    var p2 = d.packageConstraint.package;
    if (has(depsByPackage, p2)) {
      throw new Error("Can't have two dependencies on " + p2 + " in " + key);
    }
    depsByPackage[p2] = d;
  });
};

// Returns the dependencies of a (package, version), stored in a map.
// The values are Dependency objects; the key for `d` is
// `d.packageConstraint.package`.  (Don't mutate the map.)
CS.CatalogCache.prototype.getDependencyMap = function (p, v) {
  var key = pvkey(p, v);
  if (!has(this._dependencies, key)) {
    throw new Error("No entry for " + key);
  }
  return this._dependencies[key];
};

// Returns an array of version strings, sorted, possibly empty.
// (Don't mutate the result.)
CS.CatalogCache.prototype.getPackageVersions = function (pkg) {
  var result = has(this._versions, pkg) ? this._versions[pkg] : [];
  if (!result.length || result.sorted) {
    return result;
  } else {
    // sort in place, and record so that we don't sort redundantly
    // (we'll sort again if more versions are pushed onto the array)
    var pvParse = memoize(PV.parse);
    result.sort(function (a, b) {
      return PV.compare(pvParse(a), pvParse(b));
    });
    result.sorted = true;
    return result;
  }
};
CS.CatalogCache.prototype.hasPackage = function (pkg) {
  return has(this._versions, pkg);
};
CS.CatalogCache.prototype.toJSONable = function () {
  var self = this;
  var data = {};
  Object.entries(self._dependencies).forEach(function (_ref) {
    let [key, depsByPackage] = _ref;
    // depsByPackage is a map of String -> Dependency.
    // Map over the values to get an array of String.
    data[key] = Object.values(depsByPackage).map(function (dep) {
      return dep.toString();
    });
  });
  return {
    data: data
  };
};
CS.CatalogCache.fromJSONable = function (obj) {
  check(obj, {
    data: Object
  });
  var cache = new CS.CatalogCache();
  Object.entries(obj.data).forEach(function (_ref2) {
    let [pv, depsArray] = _ref2;
    check(depsArray, [String]);
    pv = CS.PackageAndVersion.fromString(pv);
    cache.addPackageVersion(pv.package, pv.version, depsArray.map(function (str) {
      return CS.Dependency.fromString(str);
    }));
  });
  return cache;
};

// Calls `iter` on each PackageAndVersion, with the second argument being
// a map from package name to Dependency.  If `iter` returns true,
// iteration is stopped.  There's no particular order to the iteration.
CS.CatalogCache.prototype.eachPackageVersion = function (iter) {
  var self = this;
  Object.keys(self._dependencies).find(function (key) {
    var stop = iter(CS.PackageAndVersion.fromString(key), self._dependencies[key]);
    return stop;
  });
};

// Calls `iter` on each package name, with the second argument being
// a list of versions present for that package (unique and sorted).
// If `iter` returns true, iteration is stopped.
ConstraintSolver.CatalogCache.prototype.eachPackage = function (iter) {
  var self = this;
  Object.keys(self._versions).find(function (key) {
    var stop = iter(key, self.getPackageVersions(key));
    return stop;
  });
};
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"catalog-loader.js":function module(require){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// packages/constraint-solver/catalog-loader.js                                                                        //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
const has = require('lodash.has');
var PV = PackageVersion;
var CS = ConstraintSolver;

// A CatalogLoader does the work of populating a CatalogCache from the
// Catalog.  When you run a unit test with canned Catalog data, there is
// a CatalogCache but no CatalogLoader.
//
// CatalogLoader acts as a minor cache layer between CatalogCache and
// the Catalog, because going to the Catalog generally means going to
// SQLite, i.e. disk, while caching a version in the CatalogCache means
// that it is available to the solver.  CatalogLoader's private cache
// allows it to over-read from the Catalog so that it can mediate
// between the granularity provided by the Catalog and the versions
// requested by the solver.
//
// We rely on the following `catalog` methods:
//
// * getSortedVersionRecords(packageName) ->
//     [{packageName, version, dependencies}]
//
//   Where `dependencies` is a map from packageName to
//   an object of the form `{ constraint: String|null,
//   references: [{arch: String, optional "weak": true}] }`.
//
// * getVersion(packageName, version) ->
//   {packageName, version, dependencies}

CS.CatalogLoader = function (fromCatalog, toCatalogCache) {
  var self = this;
  self.catalog = fromCatalog;
  self.catalogCache = toCatalogCache;
  self._sortedVersionRecordsCache = {};
};
var convertDeps = function (catalogDeps) {
  return Object.entries(catalogDeps).map(function (_ref) {
    let [pkg, dep] = _ref;
    // The dependency is strong if any of its "references"
    // (for different architectures) are strong.
    var isStrong = dep.references.some(function (ref) {
      return !ref.weak;
    });
    var constraint = dep.constraint || null;
    return new CS.Dependency(new PV.PackageConstraint(pkg, constraint), isStrong ? null : {
      isWeak: true
    });
  });
};

// Since we don't fetch different versions of a package independently
// at the moment, this helper is where we get our data.
CS.CatalogLoader.prototype._getSortedVersionRecords = async function (pkg) {
  if (!has(this._sortedVersionRecordsCache, pkg)) {
    this._sortedVersionRecordsCache[pkg] = await this.catalog.getSortedVersionRecords(pkg);
  }
  return this._sortedVersionRecordsCache[pkg];
};
CS.CatalogLoader.prototype.loadSingleVersion = async function (pkg, version) {
  var self = this;
  var cache = self.catalogCache;
  if (!cache.hasPackageVersion(pkg, version)) {
    var rec;
    if (has(self._sortedVersionRecordsCache, pkg)) {
      rec = self._sortedVersionRecordsCache[pkg].find(function (r) {
        return r.version === version;
      });
    } else {
      rec = await self.catalog.getVersion(pkg, version);
    }
    if (rec) {
      var deps = convertDeps(rec.dependencies);
      cache.addPackageVersion(pkg, version, deps);
    }
  }
};
CS.CatalogLoader.prototype.loadAllVersions = async function (pkg) {
  var self = this;
  var cache = self.catalogCache;
  var versionRecs = await self._getSortedVersionRecords(pkg);
  versionRecs.forEach(function (rec) {
    var version = rec.version;
    if (!cache.hasPackageVersion(pkg, version)) {
      var deps = convertDeps(rec.dependencies);
      cache.addPackageVersion(pkg, version, deps);
    }
  });
};

// Takes an array of package names.  Loads all versions of them and their
// (strong) dependencies.
CS.CatalogLoader.prototype.loadAllVersionsRecursive = async function (packageList) {
  var self = this;

  // Within a call to loadAllVersionsRecursive, we only visit each package
  // at most once.  If we visit a package we've already loaded, it will
  // lead to a quick scan through the versions in our cache to make sure
  // they have been loaded into the CatalogCache.
  var loadQueue = [];
  var packagesEverEnqueued = {};
  var enqueue = function (pkg) {
    if (!has(packagesEverEnqueued, pkg)) {
      packagesEverEnqueued[pkg] = true;
      loadQueue.push(pkg);
    }
  };
  packageList.forEach(enqueue);
  while (loadQueue.length) {
    var pkg = loadQueue.pop();
    await self.loadAllVersions(pkg);
    self.catalogCache.getPackageVersions(pkg).forEach(function (v) {
      var depMap = self.catalogCache.getDependencyMap(pkg, v);
      Object.entries(depMap).forEach(function (_ref2) {
        let [package2, dep] = _ref2;
        enqueue(package2);
      });
    });
  }
};
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"constraint-solver-input.js":function module(require){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// packages/constraint-solver/constraint-solver-input.js                                                               //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
const has = require('lodash.has');
const isEmpty = require('lodash.isempty');
const isEqual = require('lodash.isequal');
var PV = PackageVersion;
var CS = ConstraintSolver;

// `check` can be really slow, so this line is a valve that makes it
// easy to turn off when debugging performance problems.
var _check = check;

// The "Input" object completely specifies the input to the resolver,
// and it holds the data loaded from the Catalog as well.  It can be
// serialized to JSON and read back in for testing purposes.
CS.Input = function (dependencies, constraints, catalogCache, options) {
  var self = this;
  options = options || {};

  // PackageConstraints passed in from the tool to us (where we are a
  // uniloaded package) will have constructors that we don't recognize
  // because they come from a different copy of package-version-parser!
  // Convert them to our PackageConstraint class if necessary.  (This is
  // just top-level constraints from .meteor/packages or running from
  // checkout, so it's not a lot of data.)
  constraints = constraints.map(function (c) {
    if (c instanceof PV.PackageConstraint) {
      return c;
    } else {
      return PV.parsePackageConstraint(c.package, c.constraintString);
    }
  });

  // Note that `dependencies` and `constraints` are required (you can't
  // omit them or pass null), while the other properties have defaults.
  self.dependencies = dependencies;
  self.constraints = constraints;
  // If you add a property, make sure you add it to:
  // - The `check` statements below
  // - toJSONable (this file)
  // - fromJSONable (this file)
  // - the "input serialization" test in constraint-solver-tests.js
  // If it's an option passed in from the tool, you'll also have to
  // add it to CS.PackagesResolver#resolve.
  self.upgrade = options.upgrade || [];
  self.anticipatedPrereleases = options.anticipatedPrereleases || {};
  self.previousSolution = options.previousSolution || null;
  self.allowIncompatibleUpdate = options.allowIncompatibleUpdate || false;
  self.upgradeIndirectDepPatchVersions = options.upgradeIndirectDepPatchVersions || false;
  _check(self.dependencies, [String]);
  _check(self.constraints, [PV.PackageConstraint]);
  _check(self.upgrade, [String]);
  _check(self.anticipatedPrereleases, Match.ObjectWithValues(Match.ObjectWithValues(Boolean)));
  _check(self.previousSolution, Match.OneOf(Object, null));
  _check(self.allowIncompatibleUpdate, Boolean);
  _check(self.upgradeIndirectDepPatchVersions, Boolean);
  self.catalogCache = catalogCache;
  _check(self.catalogCache, CS.CatalogCache);
  // The catalog presumably has valid package names in it, but make sure
  // there aren't any characters in there somehow that will trip us up
  // with creating valid variable strings.
  self.catalogCache.eachPackage(function (packageName) {
    validatePackageName(packageName);
  });
  self.catalogCache.eachPackageVersion(function (packageName, depsMap) {
    Object.entries(depsMap).forEach(function (_ref) {
      let [depPackageName, deps] = _ref;
      validatePackageName(depPackageName);
    });
  });
  self.dependencies.forEach(validatePackageName);
  self.upgrade.forEach(validatePackageName);
  self.constraints.forEach(function (c) {
    validatePackageName(c.package);
  });
  if (self.previousSolution) {
    Object.keys(self.previousSolution).forEach(validatePackageName);
  }
  self._dependencySet = {}; // package name -> true
  self.dependencies.forEach(function (d) {
    self._dependencySet[d] = true;
  });
  self._upgradeSet = {};
  self.upgrade.forEach(function (u) {
    self._upgradeSet[u] = true;
  });
};
validatePackageName = function (name) {
  PV.validatePackageName(name);
  // We have some hard requirements of our own so that packages can be
  // used as solver variables.  PV.validatePackageName should already
  // enforce these requirements and more, so these checks are just a
  // backstop in case it changes under us somehow.
  if (name.charAt(0) === '$' || name.charAt(0) === '-') {
    throw new Error("First character of package name cannot be: " + name.charAt(0));
  }
  if (/ /.test(name)) {
    throw new Error("No space allowed in package name");
  }
};
CS.Input.prototype.isKnownPackage = function (p) {
  return this.catalogCache.hasPackage(p);
};
CS.Input.prototype.isRootDependency = function (p) {
  return has(this._dependencySet, p);
};
CS.Input.prototype.isUpgrading = function (p) {
  return has(this._upgradeSet, p);
};
CS.Input.prototype.isInPreviousSolution = function (p) {
  return !!(this.previousSolution && has(this.previousSolution, p));
};
function getMentionedPackages(input) {
  var packages = {}; // package -> true

  input.dependencies.forEach(function (pkg) {
    packages[pkg] = true;
  });
  input.constraints.forEach(function (constraint) {
    packages[constraint.package] = true;
  });
  if (input.previousSolution) {
    Object.entries(input.previousSolution).forEach(function (_ref2) {
      let [pkg, version] = _ref2;
      packages[pkg] = true;
    });
  }
  return Object.keys(packages);
}
CS.Input.prototype.loadFromCatalog = async function (catalogLoader) {
  // Load packages into the cache (if they aren't loaded already).
  await catalogLoader.loadAllVersionsRecursive(getMentionedPackages(this));
};
CS.Input.prototype.loadOnlyPreviousSolution = async function (catalogLoader) {
  var self = this;

  // load just the exact versions from the previousSolution
  if (self.previousSolution) {
    for (const [pkg, version] of Object.entries(self.previousSolution)) {
      await catalogLoader.loadSingleVersion(pkg, version);
    }
  }
};
CS.Input.prototype.isEqual = function (otherInput) {
  var a = this;
  var b = otherInput;

  // It would be more efficient to compare the fields directly,
  // but converting to JSON is much easier to implement.
  // This equality test is also overly sensitive to order,
  // missing opportunities to declare two inputs equal when only
  // the order has changed.

  // Omit `catalogCache` -- it's not actually part of the serialized
  // input object (it's only in `toJSONable()` for tests).
  //
  // Moreover, catalogCache is populated as-needed so their values for
  // `a` and `b` will very likely be different even if they represent
  // the same input. So by omitting `catalogCache` we no longer need
  // to reload the entire relevant part of the catalog from SQLite on
  // every rebuild!
  return isEqual(a.toJSONable(true), b.toJSONable(true));
};
CS.Input.prototype.toJSONable = function (omitCatalogCache) {
  var self = this;
  var obj = {
    dependencies: self.dependencies,
    constraints: self.constraints.map(function (c) {
      return c.toString();
    })
  };
  if (!omitCatalogCache) {
    obj.catalogCache = self.catalogCache.toJSONable();
  }

  // For readability of the resulting JSON, only include optional
  // properties that aren't the default.
  if (self.upgrade.length) {
    obj.upgrade = self.upgrade;
  }
  if (!isEmpty(self.anticipatedPrereleases)) {
    obj.anticipatedPrereleases = self.anticipatedPrereleases;
  }
  if (self.previousSolution !== null) {
    obj.previousSolution = self.previousSolution;
  }
  if (self.allowIncompatibleUpdate) {
    obj.allowIncompatibleUpdate = true;
  }
  if (self.upgradeIndirectDepPatchVersions) {
    obj.upgradeIndirectDepPatchVersions = true;
  }
  return obj;
};
CS.Input.fromJSONable = function (obj) {
  _check(obj, {
    dependencies: [String],
    constraints: [String],
    catalogCache: Object,
    anticipatedPrereleases: Match.Optional(Match.ObjectWithValues(Match.ObjectWithValues(Boolean))),
    previousSolution: Match.Optional(Match.OneOf(Object, null)),
    upgrade: Match.Optional([String]),
    allowIncompatibleUpdate: Match.Optional(Boolean),
    upgradeIndirectDepPatchVersions: Match.Optional(Boolean)
  });
  return new CS.Input(obj.dependencies, obj.constraints.map(function (cstr) {
    return PV.parsePackageConstraint(cstr);
  }), CS.CatalogCache.fromJSONable(obj.catalogCache), {
    upgrade: obj.upgrade,
    anticipatedPrereleases: obj.anticipatedPrereleases,
    previousSolution: obj.previousSolution,
    allowIncompatibleUpdate: obj.allowIncompatibleUpdate,
    upgradeIndirectDepPatchVersions: obj.upgradeIndirectDepPatchVersions
  });
};
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"version-pricer.js":function module(require){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// packages/constraint-solver/version-pricer.js                                                                        //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
const memoize = require('lodash.memoize');
var CS = ConstraintSolver;
var PV = PackageVersion;
CS.VersionPricer = function () {
  var self = this;

  // self.getVersionInfo(versionString) returns an object
  // that contains at least { major, minor, patch }.
  //
  // The VersionPricer instance stores a memoization table for
  // efficiency.
  self.getVersionInfo = memoize(PV.parse);
};
CS.VersionPricer.MODE_UPDATE = 1;
CS.VersionPricer.MODE_GRAVITY = 2;
CS.VersionPricer.MODE_GRAVITY_WITH_PATCHES = 3;

// priceVersions(versions, mode, options) calculates small integer
// costs for each version, based on whether each part of the version
// is low or high relative to the other versions with the same higher
// parts.
//
// For example, if "1.2.0" and "1.2.1" are the only 1.2.x versions
// in the versions array, they will be assigned PATCH costs of
// 1 and 0 in UPDATE mode (penalizing the older version), or 0 and 1
// in GRAVITY mode (penalizing the newer version).  When optimizing,
// the solver will prioritizing minimizing MAJOR costs, then MINOR
// costs, then PATCH costs, and then "REST" costs (which penalizing
// being old or new within versions that have the same major, minor,
// AND patch).
//
// - `versions` - Array of version strings in sorted order
// - `mode` - A MODE constant
// - `options`:
//   - `versionAfter` - if provided, the next newer version not in the
//     array but that would come next.
//   - `versionBefore` - if provided, the next older version not in the
//     the array but that would come before it.
//
// Returns: an array of 4 arrays, each of length versions.length,
// containing the MAJOR, MINOR, PATCH, and REST costs corresponding
// to the versions.
//
// MODE_UPDATE penalizes versions for being old (because we want
// them to be new), while the MODE_GRAVITY penalizes versions for
// being new (because we are trying to apply "version gravity" and
// prefer older versions).  MODE_GRAVITY_WITH_PATCHES applies gravity
// to the major and minor parts of the version, but prefers updates
// to the patch and rest of the version.
//
// Use `versionAfter` when scanning a partial array of versions
// if you want the newest version in the array to have a non-zero
// weight in MODE_UPDATE.  For example, the versions
// `["1.0.0", "1.0.1"]` will be considered to have an out-of-date
// version if versionAfter is `"2.0.0"`.  The costs returned
// won't be the same as if the whole array was scanned at once,
// but this option is useful in order to apply MODE_UPDATE to some
// versions and MODE_GRAVITY to others, for example.
//
// `versionBefore` is used in an analogous way with the GRAVITY modes.
//
// The easiest way to implement this function would be to partition
// `versions` into subarrays of versions with the same major part,
// and then partition those arrays based on the minor parts, and
// so on.  However, that's a lot of array allocations -- O(N) or
// thereabouts.  So instead we use a linear scan backwards through
// the versions array.
CS.VersionPricer.prototype.priceVersions = function (versions, mode, options) {
  var self = this;
  var getMajorMinorPatch = function (v) {
    var vInfo = self.getVersionInfo(v);
    return [vInfo.major, vInfo.minor, vInfo.patch];
  };
  var MAJOR = 0,
    MINOR = 1,
    PATCH = 2,
    REST = 3;
  var gravity; // array of MAJOR, MINOR, PATCH, REST

  switch (mode) {
    case CS.VersionPricer.MODE_UPDATE:
      gravity = [false, false, false, false];
      break;
    case CS.VersionPricer.MODE_GRAVITY:
      gravity = [true, true, true, true];
      break;
    case CS.VersionPricer.MODE_GRAVITY_WITH_PATCHES:
      gravity = [true, true, false, false];
      break;
    default:
      throw new Error("Bad mode: " + mode);
  }
  var lastMajorMinorPatch = null;
  if (options && options.versionAfter) {
    lastMajorMinorPatch = getMajorMinorPatch(options.versionAfter);
  }
  // `costs` contains arrays of whole numbers, each of which will
  // have a length of versions.length.  This is what we will return.
  var costs = [[], [], [], []]; // MAJOR, MINOR, PATCH, REST
  // How many in a row of the same MAJOR, MINOR, or PATCH have we seen?
  var countOfSame = [0, 0, 0];

  // Track how old each part of versions[i] is, in terms of how many
  // greater values there are for that part among versions with the
  // same higher parts.  For example, oldness[REST] counts the number
  // of versions after versions[i] with the same MAJOR, MINOR, and REST.
  // oldness[PATCH] counts the number of *different* higher values for
  // for PATCH among later versions with the same MAJOR and MINOR parts.
  var oldness = [0, 0, 0, 0];

  // Walk the array backwards
  for (var i = versions.length - 1; i >= 0; i--) {
    var v = versions[i];
    var majorMinorPatch = getMajorMinorPatch(v);
    if (lastMajorMinorPatch) {
      for (var k = MAJOR; k <= REST; k++) {
        if (k === REST || majorMinorPatch[k] !== lastMajorMinorPatch[k]) {
          // For the highest part that changed, bumped the oldness
          // and clear the lower oldnesses.
          oldness[k]++;
          for (var m = k + 1; m <= REST; m++) {
            if (gravity[m]) {
              // if we should actually be counting "newness" instead of
              // oldness, flip the count.  Instead of [0, 1, 1, 2, 3],
              // for example, make it [3, 2, 2, 1, 0].  This is the place
              // to do it, because we have just "closed out" a run.
              flipLastN(costs[m], countOfSame[m - 1], oldness[m]);
            }
            countOfSame[m - 1] = 0;
            oldness[m] = 0;
          }
          break;
        }
      }
    }
    for (var k = MAJOR; k <= REST; k++) {
      costs[k].push(oldness[k]);
      if (k !== REST) {
        countOfSame[k]++;
      }
    }
    lastMajorMinorPatch = majorMinorPatch;
  }
  if (options && options.versionBefore && versions.length) {
    // bump the appropriate value of oldness, as if we ran the loop
    // one more time
    majorMinorPatch = getMajorMinorPatch(options.versionBefore);
    for (var k = MAJOR; k <= REST; k++) {
      if (k === REST || majorMinorPatch[k] !== lastMajorMinorPatch[k]) {
        oldness[k]++;
        break;
      }
    }
  }

  // Flip the MAJOR costs if we have MAJOR gravity -- subtracting them
  // all from oldness[MAJOR] -- and likewise for other parts if countOfSame
  // is > 0 for the next highest part (meaning we didn't get a chance to
  // flip some of the costs because the loop ended).
  for (var k = MAJOR; k <= REST; k++) {
    if (gravity[k]) {
      flipLastN(costs[k], k === MAJOR ? costs[k].length : countOfSame[k - 1], oldness[k]);
    }
  }

  // We pushed costs onto the arrays in reverse order.  Reverse the cost
  // arrays in place before returning them.
  return [costs[MAJOR].reverse(), costs[MINOR].reverse(), costs[PATCH].reverse(), costs[REST].reverse()];
};

// "Flip" the last N elements of array in place by subtracting each
// one from `max`.  For example, if `a` is `[3,0,1,1,2]`, then calling
// `flipLastN(a, 4, 2)` mutates `a` into `[3,2,1,1,0]`.
var flipLastN = function (array, N, max) {
  var len = array.length;
  for (var i = 0; i < N; i++) {
    var j = len - 1 - i;
    array[j] = max - array[j];
  }
};

// Partition a sorted array of versions into three arrays, containing
// the versions that are `older` than the `target` version,
// `compatible` with it, or have a `higherMajor` version.
//
// For example, `["1.0.0", "2.5.0", "2.6.1", "3.0.0"]` with a target of
// `"2.5.0"` returns `{ older: ["1.0.0"], compatible: ["2.5.0", "2.6.1"],
// higherMajor: ["3.0.0"] }`.
CS.VersionPricer.prototype.partitionVersions = function (versions, target) {
  var self = this;
  var firstGteIndex = versions.length;
  var higherMajorIndex = versions.length;
  var targetVInfo = self.getVersionInfo(target);
  for (var i = 0; i < versions.length; i++) {
    var v = versions[i];
    var vInfo = self.getVersionInfo(v);
    if (firstGteIndex === versions.length && !PV.lessThan(vInfo, targetVInfo)) {
      firstGteIndex = i;
    }
    if (vInfo.major > targetVInfo.major) {
      higherMajorIndex = i;
      break;
    }
  }
  return {
    older: versions.slice(0, firstGteIndex),
    compatible: versions.slice(firstGteIndex, higherMajorIndex),
    higherMajor: versions.slice(higherMajorIndex)
  };
};

// Use a combination of calls to priceVersions with different modes in order
// to generate costs for versions relative to a "previous solution" version
// (called the "target" here).
CS.VersionPricer.prototype.priceVersionsWithPrevious = function (versions, target, takePatches) {
  var self = this;
  var parts = self.partitionVersions(versions, target);
  var result1 = self.priceVersions(parts.older, CS.VersionPricer.MODE_UPDATE, {
    versionAfter: target
  });
  // Usually, it's better to remain as close as possible to the target
  // version, but prefer higher patch versions (and wrapNums, etc.) if
  // we were passed `takePatches`.
  var result2 = self.priceVersions(parts.compatible, takePatches ? CS.VersionPricer.MODE_GRAVITY_WITH_PATCHES : CS.VersionPricer.MODE_GRAVITY);
  // If we're already bumping the major version, might as well take patches.
  var result3 = self.priceVersions(parts.higherMajor, CS.VersionPricer.MODE_GRAVITY_WITH_PATCHES,
  // not actually the version right before, but
  // gives the `major` cost the bump it needs
  {
    versionBefore: target
  });

  // Generate a fifth array, incompat, which has a 1 for each incompatible
  // version and a 0 for each compatible version.
  var incompat = [];
  var i;
  for (i = 0; i < parts.older.length; i++) {
    incompat.push(1);
  }
  for (i = 0; i < parts.compatible.length; i++) {
    incompat.push(0);
  }
  for (i = 0; i < parts.higherMajor.length; i++) {
    incompat.push(1);
  }
  return [incompat, result1[0].concat(result2[0], result3[0]), result1[1].concat(result2[1], result3[1]), result1[2].concat(result2[2], result3[2]), result1[3].concat(result2[3], result3[3])];
};
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"solver.js":function module(require){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// packages/constraint-solver/solver.js                                                                                //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
const groupBy = require('lodash.groupby');
const has = require('lodash.has');
const memoize = require('lodash.memoize');
const zip = require('lodash.zip');
var CS = ConstraintSolver;
var PV = PackageVersion;
var pvVar = function (p, v) {
  return p + ' ' + v;
};

// The "inner solver".  You construct it with a ConstraintSolver.Input object
// (which specifies the problem) and then call .getAnswer() on it.

CS.Solver = function (input, options) {
  var self = this;
  check(input, CS.Input);
  self.input = input;
  self.errors = []; // [String]

  self.pricer = new CS.VersionPricer();
  self.getConstraintFormula = memoize(_getConstraintFormula, function (p, vConstraint) {
    return p + "@" + vConstraint.raw;
  });
  self.options = options || {};
  self.Profile = self.options.Profile || CS.DummyProfile;
  self.steps = [];
  self.stepsByName = {};
  self.analysis = {};
};
CS.Solver.prototype.init = async function () {
  const self = this;
  await self.Profile.time("Solver#analyze", function () {
    return self.analyze();
  });
  self.logic = null; // Logic.Solver, initialized later
};
CS.Solver.prototype.throwAnyErrors = function () {
  if (this.errors.length) {
    var multiline = this.errors.some(function (e) {
      return /\n/.test(e);
    });
    CS.throwConstraintSolverError(this.errors.join(multiline ? '\n\n' : '\n'));
  }
};
CS.Solver.prototype.getVersions = function (pkg) {
  var self = this;
  if (has(self.analysis.allowedVersions, pkg)) {
    return self.analysis.allowedVersions[pkg];
  } else {
    return self.input.catalogCache.getPackageVersions(pkg);
  }
};

// Populates `self.analysis` with various data structures derived from the
// input.  May also throw errors, and may call methods that rely on
// analysis once that particular analysis is done (e.g. `self.getVersions`
// which relies on `self.analysis.allowedVersions`.
// TODO -> Check await Profile.time
CS.Solver.prototype.analyze = async function () {
  var self = this;
  var analysis = self.analysis;
  var input = self.input;
  var cache = input.catalogCache;
  var Profile = self.Profile;

  ////////// ANALYZE ALLOWED VERSIONS
  // (An "allowed version" is one that isn't ruled out by a top-level
  // constraint.)

  // package -> array of version strings.  If a package has an entry in
  // this map, then only the versions in the array are allowed for
  // consideration.
  analysis.allowedVersions = {};
  analysis.packagesWithNoAllowedVersions = {}; // package -> [constraints]

  // Process top-level constraints, applying them right now by
  // limiting what package versions we even consider.  This speeds up
  // solving, especially given the equality constraints on core
  // packages.  For versions we don't allow, we get to avoid generating
  // Constraint objects for their constraints, which saves us both
  // clause generation time and solver work up through the point where we
  // determine there are no conflicts between constraints.
  //
  // we can't throw any errors yet, because `input.constraints`
  // doesn't establish any dependencies (so we don't know if it's a
  // problem that some package has no legal versions), but we can
  // track such packages in packagesWithNoAllowedVersions so that we
  // throw a good error later.
  await Profile.time("analyze allowed versions", function () {
    Object.entries(groupBy(input.constraints, 'package')).forEach(function (_ref) {
      let [p, cs] = _ref;
      var versions = cache.getPackageVersions(p);
      if (!versions.length) {
        // deal with wholly unknown packages later
        return;
      }
      cs.forEach(function (constr) {
        versions = versions.filter(function (v) {
          return CS.isConstraintSatisfied(p, constr.versionConstraint, v);
        });
      });
      if (!versions.length) {
        analysis.packagesWithNoAllowedVersions[p] = cs.filter(function (c) {
          return !!c.constraintString;
        });
      }
      analysis.allowedVersions[p] = versions;
    });
  });

  ////////// ANALYZE ROOT DEPENDENCIES

  // Collect root dependencies that we've never heard of.
  analysis.unknownRootDeps = [];
  // Collect "previous solution" versions of root dependencies.
  analysis.previousRootDepVersions = [];
  await Profile.time("analyze root dependencies", function () {
    input.dependencies.forEach(function (p) {
      if (!input.isKnownPackage(p)) {
        analysis.unknownRootDeps.push(p);
      } else if (input.isInPreviousSolution(p) && !input.isUpgrading(p)) {
        analysis.previousRootDepVersions.push(new CS.PackageAndVersion(p, input.previousSolution[p]));
      }
    });

    // throw if there are unknown packages in root deps
    if (analysis.unknownRootDeps.length) {
      analysis.unknownRootDeps.forEach(function (p) {
        if (CS.isIsobuildFeaturePackage(p)) {
          self.errors.push('unsupported Isobuild feature "' + p + '" in top-level dependencies; see ' + 'https://docs.meteor.com/api/packagejs.html#isobuild-features ' + 'for a list of features and the minimum Meteor release required');
        } else {
          self.errors.push('unknown package in top-level dependencies: ' + p);
        }
      });
      self.throwAnyErrors();
    }
  });

  ////////// ANALYZE REACHABILITY

  // A "reachable" package is one that is either a root dependency or
  // a strong dependency of any "allowed" version of a reachable package.
  // In other words, we walk all strong dependencies starting
  // with the root dependencies, and visiting all allowed versions of each
  // package.
  //
  // This analysis is mainly done for performance, because if there are
  // extraneous packages in the CatalogCache (for whatever reason) we
  // want to spend as little time on them as possible.  It also establishes
  // the universe of possible "known" and "unknown" packages we might
  // come across.
  //
  // A more nuanced reachability analysis that takes versions into account
  // is probably possible.

  // package name -> true
  analysis.reachablePackages = {};
  // package name -> package versions asking for it (in pvVar form)
  analysis.unknownPackages = {};
  var markReachable = function (p) {
    analysis.reachablePackages[p] = true;
    self.getVersions(p).forEach(function (v) {
      Object.values(cache.getDependencyMap(p, v)).forEach(function (dep) {
        // `dep` is a CS.Dependency
        var p2 = dep.packageConstraint.package;
        if (!input.isKnownPackage(p2)) {
          // record this package so we will generate a variable
          // for it.  we'll try not to select it, and ultimately
          // throw an error if we are forced to.
          if (!has(analysis.unknownPackages, p2)) {
            analysis.unknownPackages[p2] = [];
          }
          analysis.unknownPackages[p2].push(pvVar(p, v));
        } else {
          if (!dep.isWeak) {
            if (!has(analysis.reachablePackages, p2)) {
              markReachable(p2);
            }
          }
        }
      });
    });
  };
  await Profile.time("analyze reachability", function () {
    input.dependencies.forEach(markReachable);
  });

  ////////// ANALYZE CONSTRAINTS

  // Array of CS.Solver.Constraint
  analysis.constraints = [];
  // packages `foo` such that there's a simple top-level equality
  // constraint about `foo`.  package name -> true.
  analysis.topLevelEqualityConstrainedPackages = {};
  await Profile.time("analyze constraints", function () {
    // Find package names with @x.y.z! overrides. We consider only
    // top-level constraints here, which includes (1) .meteor/packages,
    // (2) local package versions, and (3) Meteor release constraints.
    // Since (2) and (3) are generated programmatically without any
    // override syntax (in tools/project-context.js), the .meteor/packages
    // file is effectively the only place where override syntax has any
    // impact. This limitation is deliberate, since overriding package
    // version constraints is a power-tool that should be used sparingly
    // by application developers, and never abused by package authors.
    var overrides = new Set();
    input.constraints.forEach(function (c) {
      if (c.constraintString && c.versionConstraint.override) {
        overrides.add(c.package);
      }
    });

    // Return c.versionConstraint unless it is overridden, in which case
    // make a copy of it and set vConstraint.weakMinimum = true.
    function getVersionConstraint(c) {
      var vConstraint = c.versionConstraint;

      // The meteor-tool version can never be weakened/overridden.
      if (c.package === "meteor-tool") {
        return vConstraint;
      }

      // Overrides cannot be weakened, so in theory they could conflict
      // with each other, though that's unlikely to be a problem within a
      // single .meteor/packages file.
      if (vConstraint.override) {
        return vConstraint;
      }
      if (overrides.has(c.package)) {
        // Make a defensive shallow copy of vConstraint with the same
        // prototype (that is, PV.VersionConstraint.prototype).
        vConstraint = Object.create(Object.getPrototypeOf(vConstraint), Object.getOwnPropertyDescriptors(vConstraint));

        // This weakens the constraint so that it matches any version not
        // less than the constraint, regardless of whether the major or
        // minor versions are the same. See CS.isConstraintSatisfied in
        // constraint-solver.js for the implementation of this behavior.
        vConstraint.weakMinimum = true;
      }
      return vConstraint;
    }

    // top-level constraints
    input.constraints.forEach(function (c) {
      if (c.constraintString) {
        analysis.constraints.push(new CS.Solver.Constraint(null, c.package, getVersionConstraint(c), "constraint#" + analysis.constraints.length));
        if (c.versionConstraint.alternatives.length === 1 && c.versionConstraint.alternatives[0].type === 'exactly') {
          analysis.topLevelEqualityConstrainedPackages[c.package] = true;
        }
      }
    });

    // constraints specified in package dependencies
    Object.keys(analysis.reachablePackages).forEach(function (p) {
      self.getVersions(p).forEach(function (v) {
        var pv = pvVar(p, v);
        Object.values(cache.getDependencyMap(p, v)).forEach(function (dep) {
          // `dep` is a CS.Dependency
          var p2 = dep.packageConstraint.package;
          if (input.isKnownPackage(p2) && dep.packageConstraint.constraintString) {
            analysis.constraints.push(new CS.Solver.Constraint(pv, p2, getVersionConstraint(dep.packageConstraint), "constraint#" + analysis.constraints.length));
          }
        });
      });
    });
  });

  ////////// ANALYZE PRE-RELEASES

  await Profile.time("analyze pre-releases", function () {
    var unanticipatedPrereleases = [];
    Object.keys(analysis.reachablePackages).forEach(function (p) {
      var anticipatedPrereleases = input.anticipatedPrereleases[p];
      self.getVersions(p).forEach(function (v) {
        if (/-/.test(v) && !(anticipatedPrereleases && has(anticipatedPrereleases, v))) {
          unanticipatedPrereleases.push(pvVar(p, v));
        }
      });
    });
    analysis.unanticipatedPrereleases = unanticipatedPrereleases;
  });
};
var WholeNumber = Match.Where(Logic.isWholeNumber);

// A Step consists of a name, an array of terms, and an array of weights.
// Steps are optimized one by one.  Optimizing a Step means to find
// the minimum whole number value for the weighted sum of the terms,
// and then to enforce in the solver that the weighted sum be that number.
// Thus, when the Steps are optimized in sequence, earlier Steps take
// precedence and will stay minimized while later Steps are optimized.
//
// A term can be a package name, a package version, or any other variable
// name or Logic formula.
//
// A weight is a non-negative integer.  The weights array can be a single
// weight (which is used for all terms).
//
// The terms and weights arguments each default to [].  You can add terms
// with weights using addTerm.
//
// options is optional.
CS.Solver.Step = function (name, terms, weights) {
  check(name, String);
  terms = terms || [];
  check(terms, [String]);
  weights = weights == null ? [] : weights;
  check(weights, Match.OneOf([WholeNumber], WholeNumber));
  this.name = name;

  // mutable:
  this.terms = terms;
  this.weights = weights;
  this.optimum = null; // set when optimized
};

// If weights is a single number, you can omit the weight argument.
// Adds a term.  If weight is 0, addTerm may skip it.
CS.Solver.Step.prototype.addTerm = function (term, weight) {
  if (weight == null) {
    if (typeof this.weights !== 'number') {
      throw new Error("Must specify a weight");
    }
    weight = this.weights;
  }
  check(weight, WholeNumber);
  if (weight !== 0) {
    this.terms.push(term);
    if (typeof this.weights === 'number') {
      if (weight !== this.weights) {
        throw new Error("Can't specify a different weight now: " + weight + " != " + this.weights);
      }
    } else {
      this.weights.push(weight);
    }
  }
};
var DEBUG = false;

// Call as one of:
// * minimize(step, options)
// * minimize([step1, step2, ...], options)
// * minimize(stepName, costTerms, costWeights, options)
CS.Solver.prototype.minimize = async function (step, options) {
  var self = this;
  if (Array.isArray(step)) {
    // minimize([steps...], options)
    for (const st of step) {
      await self.minimize(st, options);
    }
    return;
  }
  if (typeof step === 'string') {
    // minimize(stepName, costTerms, costWeights, options)
    var stepName_ = arguments[0];
    var costTerms_ = arguments[1];
    var costWeights_ = arguments[2];
    var options_ = arguments[3];
    if (costWeights_ && typeof costWeights_ === 'object' && !Array.isArray(costWeights_)) {
      options_ = costWeights_;
      costWeights_ = null;
    }
    var theStep = new CS.Solver.Step(stepName_, costTerms_, costWeights_ == null ? 1 : costWeights_);
    await self.minimize(theStep, options_);
    return;
  }

  // minimize(step, options);

  await self.Profile.time("minimize " + step.name, async function () {
    var logic = self.logic;
    self.steps.push(step);
    self.stepsByName[step.name] = step;
    if (DEBUG) {
      console.log("--- MINIMIZING " + step.name);
    }
    var costWeights = step.weights;
    var costTerms = step.terms;
    var optimized = groupMutuallyExclusiveTerms(costTerms, costWeights);
    self.setSolution(await logic.minimizeWeightedSum(self.solution, optimized.costTerms, optimized.costWeights, {
      progress: async function (status, cost) {
        if (self.options.yield) {
          await self.options.yield();
        }
        if (DEBUG) {
          if (status === 'improving') {
            console.log(cost + " ... trying to improve ...");
          } else if (status === 'trying') {
            console.log("... trying " + cost + " ... ");
          }
        }
      },
      strategy: options && options.strategy
    }));
    step.optimum = self.solution.getWeightedSum(costTerms, costWeights);
    if (DEBUG) {
      console.log(step.optimum + " is optimal");
      if (step.optimum) {
        costTerms.forEach(function (t, i) {
          var w = typeof costWeights === 'number' ? costWeights : costWeights[i];
          if (w && self.solution.evaluate(t)) {
            console.log("    " + w + ": " + t);
          }
        });
      }
    }
  });
};

// This is a correctness-preserving performance optimization.
//
// Cost functions often have many terms where both the package name
// and the weight are the same.  For example, when optimizing major
// version, we might have `(foo 3.0.0)*2 + (foo 3.0.1)*2 ...`.  It's
// more efficient to give the solver `((foo 3.0.0) OR (foo 3.0.1) OR
// ...)*2 + ...`, because it separates the question of whether to use
// ANY `foo 3.x.x` variable from the question of which one.  Other
// constraints already enforce the fact that `foo 3.0.0` and `foo 3.0.1`
// are mutually exclusive variables.  We can use that fact to "relax"
// that relationship for the purposes of the weighted sum.
//
// Note that shuffling up the order of terms unnecessarily seems to
// impact performance, so it's significant that we group by package
// first, then weight, rather than vice versa.
var groupMutuallyExclusiveTerms = function (costTerms, costWeights) {
  // Return a key for a term, such that terms with the same key are
  // guaranteed to be mutually exclusive.  We assume each term is
  // a variable representing either a package or a package version.
  // We take a prefix of the variable name up to and including the
  // first space.  So "foo 1.0.0" becomes "foo " and "foo" stays "foo".
  var getTermKey = function (t) {
    var firstSpace = t.indexOf(' ');
    return firstSpace < 0 ? t : t.slice(0, firstSpace + 1);
  };

  // costWeights, as usual, may be a number or an array
  if (typeof costWeights === 'number') {
    return {
      costTerms: Object.values(groupBy(costTerms, getTermKey)).map(function (group) {
        return Logic.or(group);
      }),
      costWeights: costWeights
    };
  } else if (!costTerms.length) {
    return {
      costTerms: costTerms,
      costWeights: costWeights
    };
  } else {
    var weightedTerms = zip(costWeights, costTerms);
    var newWeightedTerms = Object.values(groupBy(weightedTerms, function (wt) {
      // construct a string from the weight and term key, for grouping
      // purposes.  since the weight comes first, there's no ambiguity
      // and the separator char could be pretty much anything.
      return wt[0] + ' ' + getTermKey(wt[1]);
    })).map(function (wts) {
      return [wts[0][0], Logic.or(wts.map(function (x) {
        return x[1];
      }))];
    });
    return {
      costTerms: newWeightedTerms.map(function (x) {
        return x[1];
      }),
      costWeights: newWeightedTerms.map(function (x) {
        return x[0];
      })
    };
  }
};

// Determine the non-zero contributions to the cost function in `step`
// based on the current solution, returning a map from term (usually
// the name of a package or package version) to positive integer cost.
CS.Solver.prototype.getStepContributions = function (step) {
  var self = this;
  var solution = self.solution;
  var contributions = {};
  var weights = step.weights;
  step.terms.forEach(function (t, i) {
    var w = typeof weights === 'number' ? weights : weights[i];
    if (w && self.solution.evaluate(t)) {
      contributions[t] = w;
    }
  });
  return contributions;
};
var addCostsToSteps = function (pkg, versions, costs, steps) {
  var pvs = versions.map(function (v) {
    return pvVar(pkg, v);
  });
  for (var j = 0; j < steps.length; j++) {
    var step = steps[j];
    var costList = costs[j];
    if (costList.length !== versions.length) {
      throw new Error("Assertion failure: Bad lengths in addCostsToSteps");
    }
    for (var i = 0; i < versions.length; i++) {
      step.addTerm(pvs[i], costList[i]);
    }
  }
};

// Get an array of "Steps" that, when minimized in order, optimizes
// the package version costs of `packages` (an array of String package
// names) according to `pricerMode`, which may be
// `CS.VersionPricer.MODE_UPDATE` or a similar mode constant.
// Wraps `VersionPricer#priceVersions`, which is tasked with calculating
// the cost of every version of every package.  This function iterates
// over `packages` and puts the result into `Step` objects.
CS.Solver.prototype.getVersionCostSteps = async function (stepBaseName, packages, pricerMode) {
  var self = this;
  var major = new CS.Solver.Step(stepBaseName + '_major');
  var minor = new CS.Solver.Step(stepBaseName + '_minor');
  var patch = new CS.Solver.Step(stepBaseName + '_patch');
  var rest = new CS.Solver.Step(stepBaseName + '_rest');
  await self.Profile.time("calculate " + stepBaseName + " version costs", function () {
    packages.forEach(function (p) {
      var versions = self.getVersions(p);
      if (versions.length >= 2) {
        var costs = self.pricer.priceVersions(versions, pricerMode);
        addCostsToSteps(p, versions, costs, [major, minor, patch, rest]);
      }
    });
  });
  return [major, minor, patch, rest];
};

// Like `getVersionCostSteps`, but wraps
// `VersionPricer#priceVersionsWithPrevious` instead of `#priceVersions`.
// The cost function is "distance" from the previous versions passed in
// as `packageAndVersion`.  (Actually it's a complicated function of the
// previous and new version.)
CS.Solver.prototype.getVersionDistanceSteps = async function (stepBaseName, packageAndVersions, takePatches) {
  var self = this;
  var incompat = new CS.Solver.Step(stepBaseName + '_incompat');
  var major = new CS.Solver.Step(stepBaseName + '_major');
  var minor = new CS.Solver.Step(stepBaseName + '_minor');
  var patch = new CS.Solver.Step(stepBaseName + '_patch');
  var rest = new CS.Solver.Step(stepBaseName + '_rest');
  await self.Profile.time("calculate " + stepBaseName + " distance costs", function () {
    packageAndVersions.forEach(function (pvArg) {
      var pkg = pvArg.package;
      var previousVersion = pvArg.version;
      var versions = self.getVersions(pkg);
      if (versions.length >= 2) {
        var costs = self.pricer.priceVersionsWithPrevious(versions, previousVersion, takePatches);
        addCostsToSteps(pkg, versions, costs, [incompat, major, minor, patch, rest]);
      }
    });
  });
  return [incompat, major, minor, patch, rest];
};
CS.Solver.prototype.currentVersionMap = function () {
  var self = this;
  var pvs = [];
  self.solution.getTrueVars().forEach(function (x) {
    if (x.indexOf(' ') >= 0) {
      // all variables with spaces in them are PackageAndVersions
      var pv = CS.PackageAndVersion.fromString(x);
      pvs.push(pv);
    }
  });
  var versionMap = {};
  pvs.forEach(function (pv) {
    if (has(versionMap, pv.package)) {
      throw new Error("Assertion failure: Selected two versions of " + pv.package + ", " + versionMap[pv.package] + " and " + pv.version);
    }
    versionMap[pv.package] = pv.version;
  });
  return versionMap;
};

// Called to re-assign `self.solution` after a call to `self.logic.solve()`,
// `solveAssuming`, or `minimize`.
CS.Solver.prototype.setSolution = function (solution) {
  var self = this;
  self.solution = solution;
  if (!self.solution) {
    throw new Error("Unexpected unsatisfiability");
  }
  // When we query a Solution, we always want to treat unknown variables
  // as "false".  Logic Solver normally throws an error if you ask it
  // to evaluate a formula containing a variable that isn't found in any
  // constraints, as a courtesy to help catch bugs, but we treat
  // variables as an open class of predicates ("foo" means package foo
  // is selected, for example), and we don't ensure that every package
  // or package version we might ask about is registered with the Solver.
  // For example, when we go to explain a conflict or generate an error
  // about an unknown package, we may ask about packages that were
  // forbidden in an early analysis of the problem and never entered
  // into the Solver.
  return self.solution.ignoreUnknownVariables();
};
CS.Solver.prototype.getAnswer = function (options) {
  var self = this;
  return self.Profile.time("Solver#getAnswer", function () {
    return self._getAnswer(options);
  });
};
CS.Solver.prototype._getAnswer = async function (options) {
  var self = this;
  var input = self.input;
  var analysis = self.analysis;
  var cache = input.catalogCache;
  var allAnswers = options && options.allAnswers; // for tests
  var Profile = self.Profile;
  var logic = await Profile.time("new Logic.Solver (MiniSat start-up)", function () {
    return new Logic.Solver();
  });
  self.logic = logic;

  // require root dependencies
  await Profile.time("require root dependencies", function () {
    input.dependencies.forEach(function (p) {
      logic.require(p);
    });
  });

  // generate package version variables for known, reachable packages
  await Profile.time("generate package variables", function () {
    Object.keys(analysis.reachablePackages).forEach(function (p) {
      if (!has(analysis.packagesWithNoAllowedVersions, p)) {
        var versionVars = self.getVersions(p).map(function (v) {
          return pvVar(p, v);
        });
        // At most one of ["foo 1.0.0", "foo 1.0.1", ...] is true.
        logic.require(Logic.atMostOne(versionVars));
        // The variable "foo" is true if and only if at least one of the
        // variables ["foo 1.0.0", "foo 1.0.1", ...] is true.
        logic.require(Logic.equiv(p, Logic.or(versionVars)));
      }
    });
  });

  // generate strong dependency requirements
  await Profile.time("generate dependency requirements", function () {
    Object.keys(analysis.reachablePackages).forEach(function (p) {
      self.getVersions(p).forEach(function (v) {
        Object.values(cache.getDependencyMap(p, v)).forEach(function (dep) {
          // `dep` is a CS.Dependency
          if (!dep.isWeak) {
            var p2 = dep.packageConstraint.package;
            logic.require(Logic.implies(pvVar(p, v), p2));
          }
        });
      });
    });
  });

  // generate constraints -- but technically don't enforce them, because
  // we haven't forced the conflictVars to be false
  await Profile.time("generate constraints", function () {
    analysis.constraints.forEach(function (c) {
      // We logically require that EITHER a constraint is marked as a
      // conflict OR it comes from a package version that is not selected
      // OR its constraint formula must be true.
      // (The constraint formula says that if toPackage is selected,
      // then a version of it that satisfies our constraint must be true.)
      logic.require(Logic.or(c.conflictVar, c.fromVar ? Logic.not(c.fromVar) : [], self.getConstraintFormula(c.toPackage, c.vConstraint)));
    });
  });

  // Establish the invariant of self.solution being a valid solution.
  // From now on, if we add some new logical requirement to the solver
  // that isn't necessarily true of `self.solution`, we must
  // recalculate `self.solution` and pass the new value to
  // self.setSolution.  It is our job to obtain the new solution in a
  // way that ensures the solution exists and doesn't put the solver
  // in an unsatisfiable state.  There are several ways to do this:
  //
  // * Calling `logic.solve()` and immediately throwing a fatal error
  //   if there's no solution (not calling `setSolution` at all)
  // * Calling `logic.solve()` in a situation where we know we have
  //   not made the problem unsatisfiable
  // * Calling `logic.solveAssuming(...)` and checking the result, only
  //   using the solution if it exists
  // * Calling `minimize()`, which always maintains satisfiability

  await Profile.time("pre-solve", function () {
    self.setSolution(logic.solve());
  });
  // There is always a solution at this point, namely,
  // select all packages (including unknown packages), select
  // any version of each known package (excluding packages with
  // "no allowed versions"), and set all conflictVars
  // to true.

  // Forbid packages with no versions allowed by top-level constraints,
  // which we didn't do earlier because we needed to establish an
  // initial solution before asking the solver if it's possible to
  // not use these packages.
  await Profile.time("forbid packages with no matching versions", async function () {
    for (const [p, constrs] of Object.entries(analysis.packagesWithNoAllowedVersions)) {
      var newSolution = await logic.solveAssuming(Logic.not(p));
      if (newSolution) {
        self.setSolution(newSolution);
        logic.forbid(p);
      } else {
        var error = 'No version of ' + p + ' satisfies all constraints: ' + constrs.map(function (constr) {
          return '@' + constr.constraintString;
        }).join(', ');
        error += '\n' + self.listConstraintsOnPackage(p);
        self.errors.push(error);
      }
    }
    self.throwAnyErrors();
  });

  // try not to use any unknown packages.  If the minimum is greater
  // than 0, we'll throw an error later, after we apply the constraints
  // and the cost function, so that we can explain the problem to the
  // user in a convincing way.
  await self.minimize('unknown_packages', Object.keys(analysis.unknownPackages));

  // try not to set the conflictVar on any constraint.  If the minimum
  // is greater than 0, we'll throw an error later, after we've run the
  // cost function, so we can show a better error.
  // If there are conflicts, this minimization can be time-consuming
  // (several seconds or more).  The strategy 'bottom-up' helps by
  // looking for solutions with few conflicts first.
  await self.minimize('conflicts', analysis.constraints.map(function (constraint) {
    return constraint.conflictVar;
  }), {
    strategy: 'bottom-up'
  });

  // Try not to use "unanticipated" prerelease versions
  await self.minimize('unanticipated_prereleases', analysis.unanticipatedPrereleases);
  var previousRootSteps = await self.getVersionDistanceSteps('previous_root', analysis.previousRootDepVersions);
  // the "previous_root_incompat" step
  var previousRootIncompat = previousRootSteps[0];
  // the "previous_root_major", "previous_root_minor", etc. steps
  var previousRootVersionParts = previousRootSteps.slice(1);
  var toUpdate = input.upgrade.filter(function (p) {
    return analysis.reachablePackages[p] === true;
  });

  // make sure packages that are being updated can still count as
  // a previous_root for the purposes of previous_root_incompat
  await Profile.time("add terms to previous_root_incompat", function () {
    toUpdate.forEach(function (p) {
      if (input.isRootDependency(p) && input.isInPreviousSolution(p)) {
        var parts = self.pricer.partitionVersions(self.getVersions(p), input.previousSolution[p]);
        parts.older.concat(parts.higherMajor).forEach(function (v) {
          previousRootIncompat.addTerm(pvVar(p, v), 1);
        });
      }
    });
  });
  if (!input.allowIncompatibleUpdate) {
    // Enforce that we don't make breaking changes to your root dependencies,
    // unless you pass --allow-incompatible-update.  It will actually be enforced
    // farther down, but for now, we want to apply this constraint before handling
    // updates.
    await self.minimize(previousRootIncompat);
  }
  await self.minimize(await self.getVersionCostSteps('update', toUpdate, CS.VersionPricer.MODE_UPDATE));
  if (input.allowIncompatibleUpdate) {
    // If you pass `--allow-incompatible-update`, we will still try to minimize
    // version changes to root deps that break compatibility, but with a lower
    // priority than taking as-new-as-possible versions for `meteor update`.
    await self.minimize(previousRootIncompat);
  }
  await self.minimize(previousRootVersionParts);
  var otherPrevious = Object.entries(input.previousSolution || []).map(function (_ref2) {
    let [p, v] = _ref2;
    return new CS.PackageAndVersion(p, v);
  }).filter(function (pv) {
    var p = pv.package;
    return analysis.reachablePackages[p] === true && !input.isRootDependency(p);
  });
  await self.minimize(await self.getVersionDistanceSteps('previous_indirect', otherPrevious, input.upgradeIndirectDepPatchVersions));
  var newRootDeps = input.dependencies.filter(function (p) {
    return !input.isInPreviousSolution(p);
  });
  await self.minimize(await self.getVersionCostSteps('new_root', newRootDeps, CS.VersionPricer.MODE_UPDATE));

  // Lock down versions of all root, previous, and updating packages that
  // are currently selected.  The reason to do this is to save the solver
  // a bunch of work (i.e. improve performance) by not asking it to
  // optimize the "unimportant" packages while also twiddling the versions
  // of the "important" packages, which would just multiply the search space.
  //
  // The important packages are root deps, packages in the previous solution,
  // and packages being upgraded.  At this point, we either have unique
  // versions for them, or else there is some kind of trade-off, like a
  // situation where raising the version of one package and lowering the
  // version of another produces the same cost -- a tie between two solutions.
  // If we have a tie, it probably won't be broken by the unimportant
  // packages, so we'll end up going with whatever we picked anyway.  (Note
  // that we have already taken the unimportant packages into account in that
  // we are only considering solutions where SOME versions can be chosen for
  // them.)  Even if optimizing the unimportant packages (coming up next)
  // was able to break a tie in the important packages, we care so little
  // about the versions of the unimportant packages that it's a very weak
  // signal.  In other words, the user might be better off with some tie-breaker
  // that looks only at the important packages anyway.
  await Profile.time("lock down important versions", function () {
    Object.entries(self.currentVersionMap()).forEach(function (_ref3) {
      let [pkg, v] = _ref3;
      if (input.isRootDependency(pkg) || input.isInPreviousSolution(pkg) || input.isUpgrading(pkg)) {
        logic.require(Logic.implies(pkg, pvVar(pkg, v)));
      }
    });
  });

  // new, indirect packages are the lowest priority
  var otherPackages = [];
  Object.keys(analysis.reachablePackages).forEach(function (p) {
    if (!(input.isRootDependency(p) || input.isInPreviousSolution(p) || input.isUpgrading(p))) {
      otherPackages.push(p);
    }
  });
  await self.minimize(await self.getVersionCostSteps('new_indirect', otherPackages, CS.VersionPricer.MODE_GRAVITY_WITH_PATCHES));
  await self.minimize('total_packages', Object.keys(analysis.reachablePackages));

  // throw errors about unknown packages
  if (self.stepsByName['unknown_packages'].optimum > 0) {
    await Profile.time("generate error for unknown packages", function () {
      var unknownPackages = Object.keys(analysis.unknownPackages);
      var unknownPackagesNeeded = unknownPackages.filter(function (p) {
        return self.solution.evaluate(p);
      });
      unknownPackagesNeeded.forEach(function (p) {
        var requirers = analysis.unknownPackages[p].filter(function (pv) {
          return self.solution.evaluate(pv);
        });
        var errorStr;
        if (CS.isIsobuildFeaturePackage(p)) {
          errorStr = 'unsupported Isobuild feature "' + p + '"; see ' + 'https://docs.meteor.com/api/packagejs.html#isobuild-features ' + 'for a list of features and the minimum Meteor release required';
        } else {
          errorStr = 'unknown package: ' + p;
        }
        requirers.forEach(function (pv) {
          errorStr += '\nRequired by: ' + pv;
        });
        self.errors.push(errorStr);
      });
    });
    self.throwAnyErrors();
  }

  // throw errors about conflicts
  if (self.stepsByName['conflicts'].optimum > 0) {
    await self.throwConflicts();
  }
  if (!input.allowIncompatibleUpdate && self.stepsByName['previous_root_incompat'].optimum > 0) {
    // we have some "incompatible root changes", where we needed to change a
    // version of a root dependency to a new version incompatible with the
    // original, but --allow-incompatible-update hasn't been passed in.
    // these are in the form of PackageAndVersion strings that we need.
    var incompatRootChanges = Object.keys(self.getStepContributions(self.stepsByName['previous_root_incompat']));
    await Profile.time("generate errors for incompatible root change", function () {
      var numActualErrors = 0;
      incompatRootChanges.forEach(function (pvStr) {
        var pv = CS.PackageAndVersion.fromString(pvStr);
        // exclude packages with top-level equality constraints (added by user
        // or by the tool pinning a version)
        if (!has(analysis.topLevelEqualityConstrainedPackages, pv.package)) {
          var prevVersion = input.previousSolution[pv.package];
          self.errors.push('Potentially incompatible change required to ' + 'top-level dependency: ' + pvStr + ', was ' + prevVersion + '.\n' + self.listConstraintsOnPackage(pv.package));
          numActualErrors++;
        }
      });
      if (numActualErrors) {
        self.errors.push('To allow potentially incompatible changes to top-level ' + 'dependencies, you must pass --allow-incompatible-update ' + 'on the command line.');
      }
    });
    self.throwAnyErrors();
  }
  var result = {
    neededToUseUnanticipatedPrereleases: self.stepsByName['unanticipated_prereleases'].optimum > 0,
    answer: await Profile.time("generate version map", function () {
      return self.currentVersionMap();
    })
  };
  if (allAnswers) {
    await Profile.time("generate all answers", function () {
      var allAnswersList = [result.answer];
      var nextAnswer = function () {
        var formula = self.solution.getFormula();
        var newSolution = logic.solveAssuming(Logic.not(formula));
        if (newSolution) {
          self.setSolution(newSolution);
          logic.forbid(formula);
        }
        return newSolution;
      };
      while (nextAnswer()) {
        allAnswersList.push(self.currentVersionMap());
      }
      result.allAnswers = allAnswersList;
    });
  }
  ;
  return result;
};

// Get a list of package-version variables that satisfy a given constraint.
var getOkVersions = function (toPackage, vConstraint, targetVersions) {
  return targetVersions.map(function (v) {
    if (CS.isConstraintSatisfied(toPackage, vConstraint, v)) {
      return pvVar(toPackage, v);
    } else {
      return null;
    }
  }).filter(Boolean);
};

// The CS.Solver constructor turns this into a memoized method.
// Memoizing the Formula object reduces clause generation a lot.
var _getConstraintFormula = function (toPackage, vConstraint) {
  var self = this;
  var targetVersions = self.getVersions(toPackage);
  var okVersions = getOkVersions(toPackage, vConstraint, targetVersions);
  if (okVersions.length === targetVersions.length) {
    return Logic.TRUE;
  } else {
    return Logic.or(Logic.not(toPackage), okVersions);
  }
};
CS.Solver.prototype.listConstraintsOnPackage = function (pkg) {
  var self = this;
  var constraints = self.analysis.constraints;
  var result = 'Constraints on package "' + pkg + '":';
  constraints.forEach(function (c) {
    if (c.toPackage === pkg) {
      var paths;
      if (c.fromVar) {
        paths = self.getPathsToPackageVersion(CS.PackageAndVersion.fromString(c.fromVar));
      } else {
        paths = [['top level']];
      }
      paths.forEach(function (path) {
        result += '\n* ' + new PV.PackageConstraint(pkg, c.vConstraint.raw) + ' <- ' + path.join(' <- ');
      });
    }
  });
  return result;
};
CS.Solver.prototype.throwConflicts = async function () {
  var self = this;
  var solution = self.solution;
  var constraints = self.analysis.constraints;
  await self.Profile.time("generate error about conflicts", function () {
    constraints.forEach(function (c) {
      // c is a CS.Solver.Constraint
      if (solution.evaluate(c.conflictVar)) {
        // skipped this constraint
        var possibleVersions = self.getVersions(c.toPackage);
        var chosenVersion = possibleVersions.find(function (v) {
          return solution.evaluate(pvVar(c.toPackage, v));
        });
        if (!chosenVersion) {
          // this can't happen, because for a constraint to be a problem,
          // we must have chosen some version of the package it applies to!
          throw new Error("Internal error: Version not found");
        }
        var error = 'Conflict: Constraint ' + new PV.PackageConstraint(c.toPackage, c.vConstraint) + ' is not satisfied by ' + c.toPackage + ' ' + chosenVersion + '.';
        error += '\n' + self.listConstraintsOnPackage(c.toPackage);

        // Avoid printing exactly the same error twice.  eg, if we have two
        // different packages which have the same unsatisfiable constraint.
        if (self.errors.indexOf(error) === -1) {
          self.errors.push(error);
        }
      }
    });
  });

  // always throws, never returns
  self.throwAnyErrors();
  throw new Error("Internal error: conflicts could not be explained");
};

// Takes a PackageVersion and returns an array of arrays of PackageVersions.
// If the `packageVersion` is not selected in `self.solution`, returns
// an empty array.  Otherwise, returns an array of all paths from
// root dependencies to the package, in reverse order.  In other words,
// the first element of each path is `packageVersion`,
// and the last element is the selected version of a root dependency.
//
// Ok, it isn't all paths.  Because that would be crazy (combinatorial
// explosion).  It stops at root dependencies and tries to filter out
// ones that are definitely longer than another.
CS.Solver.prototype.getPathsToPackageVersion = function (packageAndVersion) {
  check(packageAndVersion, CS.PackageAndVersion);
  var self = this;
  var input = self.input;
  var cache = input.catalogCache;
  var solution = self.solution;
  var versionMap = self.currentVersionMap();
  var hasDep = function (p1, p2) {
    // Include weak dependencies, because their constraints matter.
    return has(cache.getDependencyMap(p1, versionMap[p1]), p2);
  };
  var allPackages = Object.keys(versionMap);
  var getPaths = function (pv, _ignorePackageSet) {
    if (!solution.evaluate(pv.toString())) {
      return [];
    }
    var pkg = pv.package;
    if (input.isRootDependency(pkg)) {
      return [[pv]];
    }
    var newIgnorePackageSet = Object.assign({}, _ignorePackageSet);
    newIgnorePackageSet[pkg] = true;
    var paths = [];
    var shortestLength = null;
    allPackages.forEach(function (p) {
      if (!has(newIgnorePackageSet, p) && solution.evaluate(p) && hasDep(p, pkg)) {
        var newPV = new CS.PackageAndVersion(p, versionMap[p]);
        getPaths(newPV, newIgnorePackageSet).forEach(function (path) {
          var newPath = [pv].concat(path);
          if (!paths.length || newPath.length < shortestLength) {
            paths.push(newPath);
            shortestLength = newPath.length;
          }
        });
      }
    });
    return paths;
  };
  return getPaths(packageAndVersion, {});
};
CS.Solver.Constraint = function (fromVar, toPackage, vConstraint, conflictVar) {
  this.fromVar = fromVar;
  this.toPackage = toPackage;
  this.vConstraint = vConstraint;
  this.conflictVar = conflictVar;

  // this.fromVar is a return value of pvVar(p, v), or null for a
  // top-level constraint
  check(this.fromVar, Match.OneOf(String, null));
  check(this.toPackage, String); // package name
  check(this.vConstraint, PV.VersionConstraint);
  check(this.conflictVar, String);
};
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"constraint-solver.js":function module(require){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// packages/constraint-solver/constraint-solver.js                                                                     //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
const isEqual = require('lodash.isequal');
var PV = PackageVersion;
var CS = ConstraintSolver;

// This is the entry point for the constraint-solver package.  The tool
// creates a ConstraintSolver.PackagesResolver and calls .resolve on it.

CS.PackagesResolver = function (catalog, options) {
  var self = this;
  self.catalog = catalog;
  self.catalogCache = new CS.CatalogCache();
  self.catalogLoader = new CS.CatalogLoader(self.catalog, self.catalogCache);
  self._options = {
    nudge: options && options.nudge,
    yield: options && options.yield,
    Profile: options && options.Profile,
    // For resultCache, pass in an empty object `{}`, and PackagesResolver
    // will put data on it.  Pass in the same object again to allow reusing
    // the result from the previous run.
    resultCache: options && options.resultCache
  };
};

// dependencies - an array of string names of packages (not slices)
// constraints - an array of PV.PackageConstraints
// options:
//  - upgrade - list of dependencies for which upgrade is prioritized higher
//    than keeping the old version
//  - previousSolution - mapping from package name to a version that was used in
//    the previous constraint solver run
//  - anticipatedPrereleases: mapping from package name to version to true;
//    included versions are the only pre-releases that are allowed to match
//    constraints that don't specifically name them during the "try not to
//    use unanticipated pre-releases" pass
//  - allowIncompatibleUpdate: allows choosing versions of
//    root dependencies that are incompatible with the previous solution,
//    if necessary to satisfy all constraints
//  - upgradeIndirectDepPatchVersions: also upgrade indirect dependencies
//    to newer patch versions, proactively
//  - missingPreviousVersionIsError - throw an error if a package version in
//    previousSolution is not found in the catalog
//  - supportedIsobuildFeaturePackages - map from package name to list of
//    version strings of isobuild feature packages that are available in the
//    catalog
CS.PackagesResolver.prototype.resolve = async function (dependencies, constraints, options) {
  var self = this;
  options = options || {};
  var Profile = self._options.Profile || CS.DummyProfile;
  var input = await Profile.time("new CS.Input", function () {
    const {
      upgrade,
      anticipatedPrereleases,
      previousSolution,
      allowIncompatibleUpdate,
      upgradeIndirectDepPatchVersions
    } = options;
    return new CS.Input(dependencies, constraints, self.catalogCache, {
      upgrade,
      anticipatedPrereleases,
      previousSolution,
      allowIncompatibleUpdate,
      upgradeIndirectDepPatchVersions
    });
  });

  // The constraint solver avoids re-solving everything from scratch on
  // rebuilds if the current input of top-level constraints matches the
  // previously solved input (also just top-level constraints). This is
  // slightly unsound, because non-top-level dependency constraints might
  // have changed, but it's important for performance, and relatively
  // harmless in practice (if there's a version conflict, you'll find out
  // about it the next time you do a full restart of the development
  // server). The unsoundness can cause problems for tests, however, so it
  // may be a good idea to set this environment variable to "true" to
  // disable the caching entirely.
  const disableCaching = !!JSON.parse(process.env.METEOR_DISABLE_CONSTRAINT_SOLVER_CACHING || "false");
  let resultCache = self._options.resultCache;
  if (disableCaching) {
    resultCache = null;
  } else if (resultCache && resultCache.lastInput && isEqual(resultCache.lastInput, input.toJSONable(true))) {
    return resultCache.lastOutput;
  }
  if (options.supportedIsobuildFeaturePackages) {
    Object.entries(options.supportedIsobuildFeaturePackages).forEach(function (_ref) {
      let [pkg, versions] = _ref;
      versions.forEach(function (version) {
        input.catalogCache.addPackageVersion(pkg, version, []);
      });
    });
  }
  await Profile.time("Input#loadOnlyPreviousSolution", function () {
    return input.loadOnlyPreviousSolution(self.catalogLoader);
  });
  if (options.previousSolution && options.missingPreviousVersionIsError) {
    // see comment where missingPreviousVersionIsError is passed in
    await Profile.time("check for previous versions in catalog", function () {
      Object.entries(options.previousSolution).forEach(function (_ref2) {
        let [pkg, version] = _ref2;
        if (!input.catalogCache.hasPackageVersion(pkg, version)) {
          CS.throwConstraintSolverError("Package version not in catalog: " + pkg + " " + version);
        }
      });
    });
  }
  var resolveOptions = {
    nudge: self._options.nudge,
    yield: self._options.yield,
    Profile: self._options.Profile
  };
  var output = null;
  if (options.previousSolution && !input.upgrade && !input.upgradeIndirectDepPatchVersions) {
    // Try solving first with just the versions from previousSolution in
    // the catalogCache, so that we don't have to solve the big problem
    // if we don't have to. But don't do this if we're attempting to upgrade
    // packages, because that would always result in just using the current
    // version, hence disabling upgrades.
    try {
      output = await CS.PackagesResolver._resolveWithInput(input, resolveOptions);
    } catch (e) {
      if (e.constraintSolverError) {
        output = null;
      } else {
        throw e;
      }
    }
  }
  if (!output) {
    // do a solve with all package versions available in the catalog.
    await Profile.time("Input#loadFromCatalog", function () {
      return input.loadFromCatalog(self.catalogLoader);
    });

    // if we fail to find a solution this time, this will throw.
    output = await CS.PackagesResolver._resolveWithInput(input, resolveOptions);
  }
  if (resultCache) {
    resultCache.lastInput = input.toJSONable(true);
    resultCache.lastOutput = output;
  }
  return output;
};

// Exposed for tests.
//
// Options (all optional):
// - nudge (function to be called when possible to "nudge" the progress spinner)
// - allAnswers (for testing, calculate all possible answers and put an extra
//   property named "allAnswers" on the result)
// - Profile (the profiler interface in `tools/profile.js`)
CS.PackagesResolver._resolveWithInput = async function (input, options) {
  options = options || {};
  if (Meteor.isServer && process.env['METEOR_PRINT_CONSTRAINT_SOLVER_INPUT']) {
    console.log("CONSTRAINT_SOLVER_INPUT = ");
    console.log(JSON.stringify(input.toJSONable(), null, 2));
  }
  var solver = await (options.Profile || CS.DummyProfile).time("new CS.Solver", async function () {
    const _solver = new CS.Solver(input, {
      nudge: options.nudge,
      yield: options.yield,
      Profile: options.Profile
    });
    await _solver.init();
    return _solver;
  });

  // Disable runtime type checks (they slow things down a bunch)
  return await Logic.disablingAssertions(function () {
    // if we're here, no conflicts were found (or an error would have
    // been thrown)
    return solver.getAnswer({
      allAnswers: options.allAnswers
    });
  });
};

// - package: String package name
// - vConstraint: a PackageVersion.VersionConstraint, or an object
//   with an `alternatives` property lifted from one.
// - version: version String
CS.isConstraintSatisfied = function (pkg, vConstraint, version) {
  return vConstraint.alternatives.some(function (simpleConstraint) {
    var type = simpleConstraint.type;
    if (type === "any-reasonable") {
      return true;
    }

    // If any top-level constraints use the @x.y.z! override syntax, all
    // other constraints on the same package will be marked with the
    // weakMinimum property, which means they constrain nothing other than
    // the minimum version of the package. Look for weakMinimum in the
    // CS.Solver#analyze method for related logic.
    if (vConstraint.weakMinimum) {
      return !PV.lessThan(PV.parse(version), PV.parse(simpleConstraint.versionString));
    }
    if (type === "exactly") {
      var cVersion = simpleConstraint.versionString;
      return cVersion === version;
    }
    if (type === 'compatible-with') {
      if (typeof simpleConstraint.test === "function") {
        return simpleConstraint.test(version);
      }
      var cv = PV.parse(simpleConstraint.versionString);
      var v = PV.parse(version);

      // If the candidate version is less than the version named in the
      // constraint, we are not satisfied.
      if (PV.lessThan(v, cv)) {
        return false;
      }

      // To be compatible, the two versions must have the same major version
      // number.
      if (v.major !== cv.major) {
        return false;
      }
      return true;
    }
    throw Error("Unknown constraint type: " + type);
  });
};
CS.throwConstraintSolverError = function (message) {
  var e = new Error(message);
  e.constraintSolverError = true;
  throw e;
};

// This function is duplicated in tools/compiler.js.
CS.isIsobuildFeaturePackage = function (packageName) {
  return /^isobuild:/.test(packageName);
};

// Implements the Profile interface (as we use it) but doesn't do
// anything.
CS.DummyProfile = function (bucket, f) {
  return f;
};
CS.DummyProfile.time = function (bucket, f) {
  return f();
};
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"node_modules":{"lodash.has":{"package.json":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.has/package.json                                          //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.exports = {
  "name": "lodash.has",
  "version": "4.5.2"
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"index.js":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.has/index.js                                              //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.useNode();
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

}},"lodash.memoize":{"package.json":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.memoize/package.json                                      //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.exports = {
  "name": "lodash.memoize",
  "version": "4.1.2"
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"index.js":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.memoize/index.js                                          //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.useNode();
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

}},"lodash.isempty":{"package.json":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.isempty/package.json                                      //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.exports = {
  "name": "lodash.isempty",
  "version": "4.4.0"
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"index.js":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.isempty/index.js                                          //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.useNode();
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

}},"lodash.isequal":{"package.json":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.isequal/package.json                                      //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.exports = {
  "name": "lodash.isequal",
  "version": "4.5.0"
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"index.js":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.isequal/index.js                                          //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.useNode();
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

}},"lodash.groupby":{"package.json":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.groupby/package.json                                      //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.exports = {
  "name": "lodash.groupby",
  "version": "4.6.0"
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"index.js":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.groupby/index.js                                          //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.useNode();
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

}},"lodash.zip":{"package.json":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.zip/package.json                                          //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.exports = {
  "name": "lodash.zip",
  "version": "4.2.0"
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"index.js":function module(require,exports,module){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                     //
// node_modules/meteor/constraint-solver/node_modules/lodash.zip/index.js                                              //
//                                                                                                                     //
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                       //
module.useNode();
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

}}}}}}},{
  "extensions": [
    ".js",
    ".json"
  ]
});


/* Exports */
return {
  export: function () { return {
      ConstraintSolver: ConstraintSolver
    };},
  require: require,
  eagerModulePaths: [
    "/node_modules/meteor/constraint-solver/datatypes.js",
    "/node_modules/meteor/constraint-solver/catalog-cache.js",
    "/node_modules/meteor/constraint-solver/catalog-loader.js",
    "/node_modules/meteor/constraint-solver/constraint-solver-input.js",
    "/node_modules/meteor/constraint-solver/version-pricer.js",
    "/node_modules/meteor/constraint-solver/solver.js",
    "/node_modules/meteor/constraint-solver/constraint-solver.js"
  ]
}});

//# sourceURL=meteor://💻app/packages/constraint-solver.js
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
