Package["core-runtime"].queue("caching-compiler",function () {/* Imports */
var Meteor = Package.meteor.Meteor;
var global = Package.meteor.global;
var meteorEnv = Package.meteor.meteorEnv;
var EmitterPromise = Package.meteor.EmitterPromise;
var ECMAScript = Package.ecmascript.ECMAScript;
var Random = Package.random.Random;
var meteorInstall = Package.modules.meteorInstall;
var Promise = Package.promise.Promise;

/* Package-scope variables */
var CachingCompilerBase, CachingCompiler, MultiFileCachingCompiler;

var require = meteorInstall({"node_modules":{"meteor":{"caching-compiler":{"caching-compiler.js":function module(require){

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                  //
// packages/caching-compiler/caching-compiler.js                                                                    //
//                                                                                                                  //
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                    //
const fs = Plugin.fs;
const path = Plugin.path;
const createHash = Npm.require('crypto').createHash;
const assert = Npm.require('assert');
const LRUCache = Npm.require('lru-cache');

// Base class for CachingCompiler and MultiFileCachingCompiler.
CachingCompilerBase = class CachingCompilerBase {
  constructor(_ref) {
    let {
      compilerName,
      defaultCacheSize,
      maxParallelism = 20
    } = _ref;
    this._compilerName = compilerName;
    this._maxParallelism = maxParallelism;
    const compilerNameForEnvar = compilerName.toUpperCase().replace('/-/g', '_').replace(/[^A-Z0-9_]/g, '');
    const envVarPrefix = 'METEOR_' + compilerNameForEnvar + '_CACHE_';
    const debugEnvVar = envVarPrefix + 'DEBUG';
    this._cacheDebugEnabled = !!process.env[debugEnvVar];
    const cacheSizeEnvVar = envVarPrefix + 'SIZE';
    this._cacheSize = +process.env[cacheSizeEnvVar] || defaultCacheSize;
    this._diskCache = null;

    // For testing.
    this._callCount = 0;

    // Callbacks that will be called after the linker is done processing
    // files, after all lazy compilation has finished.
    this._afterLinkCallbacks = [];
  }

  // Your subclass must override this method to define the key used to identify
  // a particular version of an InputFile.
  //
  // Given an InputFile (the data type passed to processFilesForTarget as part
  // of the Plugin.registerCompiler API), returns a cache key that represents
  // it. This cache key can be any JSON value (it will be converted internally
  // into a hash).  This should reflect any aspect of the InputFile that affects
  // the output of `compileOneFile`. Typically you'll want to include
  // `inputFile.getDeclaredExports()`, and perhaps
  // `inputFile.getPathInPackage()` or `inputFile.getDeclaredExports` if
  // `compileOneFile` pays attention to them.
  //
  // Note that for MultiFileCachingCompiler, your cache key doesn't need to
  // include the file's path, because that is automatically taken into account
  // by the implementation. CachingCompiler subclasses can choose whether or not
  // to include the file's path in the cache key.
  getCacheKey(inputFile) {
    throw Error('CachingCompiler subclass should implement getCacheKey!');
  }

  // Your subclass must override this method to define how a CompileResult
  // translates into adding assets to the bundle.
  //
  // This method is given an InputFile (the data type passed to
  // processFilesForTarget as part of the Plugin.registerCompiler API) and a
  // CompileResult (either returned directly from compileOneFile or read from
  // the cache).  It should call methods like `inputFile.addJavaScript`
  // and `inputFile.error`.
  addCompileResult(inputFile, compileResult) {
    throw Error('CachingCompiler subclass should implement addCompileResult!');
  }

  // Your subclass must override this method to define the size of a
  // CompilerResult (used by the in-memory cache to limit the total amount of
  // data cached).
  compileResultSize(compileResult) {
    throw Error('CachingCompiler subclass should implement compileResultSize!');
  }

  // Your subclass may override this method to define an alternate way of
  // stringifying CompilerResults.  Takes a CompileResult and returns a string.
  stringifyCompileResult(compileResult) {
    return JSON.stringify(compileResult);
  }
  // Your subclass may override this method to define an alternate way of
  // parsing CompilerResults from string.  Takes a string and returns a
  // CompileResult.  If the string doesn't represent a valid CompileResult, you
  // may want to return null instead of throwing, which will make
  // CachingCompiler ignore the cache.
  parseCompileResult(stringifiedCompileResult) {
    return this._parseJSONOrNull(stringifiedCompileResult);
  }
  _parseJSONOrNull(json) {
    try {
      return JSON.parse(json);
    } catch (e) {
      if (e instanceof SyntaxError) return null;
      throw e;
    }
  }
  _cacheDebug(message) {
    if (!this._cacheDebugEnabled) return;
    console.log("CACHE(".concat(this._compilerName, "): ").concat(message));
  }
  setDiskCacheDirectory(diskCache) {
    if (this._diskCache) throw Error('setDiskCacheDirectory called twice?');
    this._diskCache = diskCache;
  }

  // Since so many compilers will need to calculate the size of a SourceMap in
  // their compileResultSize, this method is provided.
  sourceMapSize(sm) {
    if (!sm) return 0;
    // sum the length of sources and the mappings, the size of
    // metadata is ignored, but it is not a big deal
    return sm.mappings.length + (sm.sourcesContent || []).reduce(function (soFar, current) {
      return soFar + (current ? current.length : 0);
    }, 0);
  }

  // Called by the compiler plugins system after all linking and lazy
  // compilation has finished.
  async afterLink() {
    for (const callback of this._afterLinkCallbacks.splice(0)) {
      await callback();
    }
  }

  // Borrowed from another MIT-licensed project that benjamn wrote:
  // https://github.com/reactjs/commoner/blob/235d54a12c/lib/util.js#L136-L168
  _deepHash(val) {
    const hash = createHash('sha1');
    let type = typeof val;
    if (val === null) {
      type = 'null';
    }
    hash.update(type + '\0');
    switch (type) {
      case 'object':
        const keys = Object.keys(val);

        // Array keys will already be sorted.
        if (!Array.isArray(val)) {
          keys.sort();
        }
        keys.forEach(key => {
          if (typeof val[key] === 'function') {
            // Silently ignore nested methods, but nevertheless complain below
            // if the root value is a function.
            return;
          }
          hash.update(key + '\0').update(this._deepHash(val[key]));
        });
        break;
      case 'function':
        assert.ok(false, 'cannot hash function objects');
        break;
      default:
        hash.update('' + val);
        break;
    }
    return hash.digest('hex');
  }

  // Write the file atomically.
  _writeFile(filename, contents) {
    const tempFilename = filename + '.tmp.' + Random.id();
    try {
      fs.writeFileSync(tempFilename, contents);
      fs.renameSync(tempFilename, filename);
    } catch (e) {
      // ignore errors, it's just a cache
      this._cacheDebug(e);
    }
  }

  // Helper function. Returns the body of the file as a string, or null if it
  // doesn't exist.
  _readFileOrNull(filename) {
    try {
      return fs.readFileSync(filename, 'utf8');
    } catch (e) {
      if (e && e.code === 'ENOENT') return null;
      throw e;
    }
  }
};

// CachingCompiler is a class designed to be used with Plugin.registerCompiler
// which implements in-memory and on-disk caches for the files that it
// processes.  You should subclass CachingCompiler and define the following
// methods: getCacheKey, compileOneFile, addCompileResult, and
// compileResultSize.
//
// CachingCompiler assumes that files are processed independently of each other;
// there is no 'import' directive allowing one file to reference another.  That
// is, editing one file should only require that file to be rebuilt, not other
// files.
//
// The data that is cached for each file is of a type that is (implicitly)
// defined by your subclass. CachingCompiler refers to this type as
// `CompileResult`, but this isn't a single type: it's up to your subclass to
// decide what type of data this is.  You should document what your subclass's
// CompileResult type is.
//
// Your subclass's compiler should call the superclass compiler specifying the
// compiler name (used to generate environment variables for debugging and
// tweaking in-memory cache size) and the default cache size.
//
// By default, CachingCompiler processes each file in "parallel". That is, if it
// needs to yield to read from the disk cache, or if getCacheKey,
// compileOneFile, or addCompileResult yields, it will start processing the next
// few files. To set how many files can be processed in parallel (including
// setting it to 1 if your subclass doesn't support any parallelism), pass the
// maxParallelism option to the superclass constructor.
//
// For example (using ES2015 via the ecmascript package):
//
//   class AwesomeCompiler extends CachingCompiler {
//     constructor() {
//       super({
//         compilerName: 'awesome',
//         defaultCacheSize: 1024*1024*10,
//       });
//     }
//     // ... define the other methods
//   }
//   Plugin.registerCompile({
//     extensions: ['awesome'],
//   }, () => new AwesomeCompiler());
//
// XXX maybe compileResultSize and stringifyCompileResult should just be methods
// on CompileResult? Sort of hard to do that with parseCompileResult.
CachingCompiler = class CachingCompiler extends CachingCompilerBase {
  constructor(_ref2) {
    let {
      compilerName,
      defaultCacheSize,
      maxParallelism = 20
    } = _ref2;
    super({
      compilerName,
      defaultCacheSize,
      maxParallelism
    });

    // Maps from a hashed cache key to a compileResult.
    this._cache = new LRUCache({
      max: this._cacheSize,
      length: value => this.compileResultSize(value)
    });
  }

  // Your subclass must override this method to define the transformation from
  // InputFile to its cacheable CompileResult).
  //
  // Given an InputFile (the data type passed to processFilesForTarget as part
  // of the Plugin.registerCompiler API), compiles the file and returns a
  // CompileResult (the cacheable data type specific to your subclass).
  //
  // This method is not called on files when a valid cache entry exists in
  // memory or on disk.
  //
  // On a compile error, you should call `inputFile.error` appropriately and
  // return null; this will not be cached.
  //
  // This method should not call `inputFile.addJavaScript` and similar files!
  // That's what addCompileResult is for.
  compileOneFile(inputFile) {
    throw Error('CachingCompiler subclass should implement compileOneFile!');
  }

  // The processFilesForTarget method from the Plugin.registerCompiler API. If
  // you have processing you want to perform at the beginning or end of a
  // processing phase, you may want to override this method and call the
  // superclass implementation from within your method.
  async processFilesForTarget(inputFiles) {
    const cacheMisses = [];
    const arches = this._cacheDebugEnabled && Object.create(null);
    for (const inputFile of inputFiles) {
      if (arches) {
        arches[inputFile.getArch()] = 1;
      }
      const getResult = async () => {
        const cacheKey = this._deepHash(this.getCacheKey(inputFile));
        let compileResult = this._cache.get(cacheKey);
        if (!compileResult) {
          compileResult = this._readCache(cacheKey);
          if (compileResult) {
            this._cacheDebug("Loaded ".concat(inputFile.getDisplayPath()));
          }
        }
        if (!compileResult) {
          cacheMisses.push(inputFile.getDisplayPath());
          compileResult = await this.compileOneFile(inputFile);
          if (!compileResult) {
            // compileOneFile should have called inputFile.error.
            //  We don't cache failures for now.
            return;
          }

          // Save what we've compiled.
          this._cache.set(cacheKey, compileResult);
          this._writeCacheAsync(cacheKey, compileResult);
        }
        return compileResult;
      };
      if (this.compileOneFileLater && inputFile.supportsLazyCompilation) {
        await this.compileOneFileLater(inputFile, getResult);
      } else {
        const result = await getResult();
        if (result) {
          await this.addCompileResult(inputFile, result);
        }
      }
    }
    if (this._cacheDebugEnabled) {
      this._afterLinkCallbacks.push(() => {
        cacheMisses.sort();
        this._cacheDebug("Ran (#".concat(++this._callCount, ") on: ").concat(JSON.stringify(cacheMisses), " ").concat(JSON.stringify(Object.keys(arches).sort())));
      });
    }
  }
  _cacheFilename(cacheKey) {
    // We want cacheKeys to be hex so that they work on any FS and never end in
    // .cache.
    if (!/^[a-f0-9]+$/.test(cacheKey)) {
      throw Error('bad cacheKey: ' + cacheKey);
    }
    return path.join(this._diskCache, cacheKey + '.cache');
  }
  // Load a cache entry from disk. Returns the compileResult object
  // and loads it into the in-memory cache too.
  _readCache(cacheKey) {
    if (!this._diskCache) {
      return null;
    }
    const cacheFilename = this._cacheFilename(cacheKey);
    const compileResult = this._readAndParseCompileResultOrNull(cacheFilename);
    if (!compileResult) {
      return null;
    }
    this._cache.set(cacheKey, compileResult);
    return compileResult;
  }
  _writeCacheAsync(cacheKey, compileResult) {
    if (!this._diskCache) return;
    const cacheFilename = this._cacheFilename(cacheKey);
    const cacheContents = this.stringifyCompileResult(compileResult);
    this._writeFile(cacheFilename, cacheContents);
  }

  // Returns null if the file does not exist or can't be parsed; otherwise
  // returns the parsed compileResult in the file.
  _readAndParseCompileResultOrNull(filename) {
    const raw = this._readFileOrNull(filename);
    return this.parseCompileResult(raw);
  }
};
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

},"multi-file-caching-compiler.js":function module(require,exports,module){

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                                                  //
// packages/caching-compiler/multi-file-caching-compiler.js                                                         //
//                                                                                                                  //
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
                                                                                                                    //
!function (module1) {
  const path = Plugin.path;
  const LRUCache = Npm.require('lru-cache');

  // MultiFileCachingCompiler is like CachingCompiler, but for implementing
  // languages which allow files to reference each other, such as CSS
  // preprocessors with `@import` directives.
  //
  // Like CachingCompiler, you should subclass MultiFileCachingCompiler and define
  // the following methods: getCacheKey, compileOneFile, addCompileResult, and
  // compileResultSize.  compileOneFile gets an additional allFiles argument and
  // returns an array of referenced import paths in addition to the CompileResult.
  // You may also override isRoot and getAbsoluteImportPath to customize
  // MultiFileCachingCompiler further.
  MultiFileCachingCompiler = class MultiFileCachingCompiler extends CachingCompilerBase {
    constructor(_ref) {
      let {
        compilerName,
        defaultCacheSize,
        maxParallelism
      } = _ref;
      super({
        compilerName,
        defaultCacheSize,
        maxParallelism
      });

      // Maps from cache key to { compileResult, cacheKeys }, where
      // cacheKeys is an object mapping from absolute import path to hashed
      // cacheKey for each file referenced by this file (including itself).
      this._cache = new LRUCache({
        max: this._cacheSize,
        // We ignore the size of cacheKeys here.
        length: value => this.compileResultSize(value.compileResult)
      });
    }

    // Your subclass must override this method to define the transformation from
    // InputFile to its cacheable CompileResult).
    //
    // Arguments:
    //   - inputFile is the InputFile to process
    //   - allFiles is a a Map mapping from absolute import path to InputFile of
    //     all files being processed in the target
    // Returns an object with keys:
    //   - compileResult: the CompileResult (the cacheable data type specific to
    //     your subclass).
    //   - referencedImportPaths: an array of absolute import paths of files
    //     which were refererenced by the current file.  The current file
    //     is included implicitly.
    //
    // This method is not called on files when a valid cache entry exists in
    // memory or on disk.
    //
    // On a compile error, you should call `inputFile.error` appropriately and
    // return null; this will not be cached.
    //
    // This method should not call `inputFile.addJavaScript` and similar files!
    // That's what addCompileResult is for.
    compileOneFile(inputFile, allFiles) {
      throw Error('MultiFileCachingCompiler subclass should implement compileOneFile!');
    }

    // Your subclass may override this to declare that a file is not a "root" ---
    // ie, it can be included from other files but is not processed on its own. In
    // this case, MultiFileCachingCompiler won't waste time trying to look for a
    // cache for its compilation on disk.
    isRoot(inputFile) {
      return true;
    }

    // Returns the absolute import path for an InputFile. By default, this is a
    // path is a path of the form "{package}/path/to/file" for files in packages
    // and "{}/path/to/file" for files in apps. Your subclass may override and/or
    // call this method.
    getAbsoluteImportPath(inputFile) {
      if (inputFile.getPackageName() === null) {
        return '{}/' + inputFile.getPathInPackage();
      }
      return '{' + inputFile.getPackageName() + '}/' + inputFile.getPathInPackage();
    }

    // The processFilesForTarget method from the Plugin.registerCompiler API.
    async processFilesForTarget(inputFiles) {
      const allFiles = new Map();
      const cacheKeyMap = new Map();
      const cacheMisses = [];
      const arches = this._cacheDebugEnabled && Object.create(null);
      inputFiles.forEach(inputFile => {
        const importPath = this.getAbsoluteImportPath(inputFile);
        allFiles.set(importPath, inputFile);
        cacheKeyMap.set(importPath, this._getCacheKeyWithPath(inputFile));
      });
      for (const inputFile of inputFiles) {
        if (arches) {
          arches[inputFile.getArch()] = 1;
        }
        const getResult = async () => {
          const absoluteImportPath = this.getAbsoluteImportPath(inputFile);
          const cacheKey = cacheKeyMap.get(absoluteImportPath);
          let cacheEntry = this._cache.get(cacheKey);
          if (!cacheEntry) {
            cacheEntry = this._readCache(cacheKey);
            if (cacheEntry) {
              this._cacheDebug("Loaded ".concat(absoluteImportPath));
            }
          }
          if (!(cacheEntry && this._cacheEntryValid(cacheEntry, cacheKeyMap))) {
            cacheMisses.push(inputFile.getDisplayPath());
            const compileOneFileReturn = await this.compileOneFile(inputFile, allFiles);
            if (!compileOneFileReturn) {
              // compileOneFile should have called inputFile.error.
              // We don't cache failures for now.
              return;
            }
            const {
              compileResult,
              referencedImportPaths
            } = compileOneFileReturn;
            cacheEntry = {
              compileResult,
              cacheKeys: {
                // Include the hashed cache key of the file itself...
                [absoluteImportPath]: cacheKeyMap.get(absoluteImportPath)
              }
            };

            // ... and of the other referenced files.
            referencedImportPaths.forEach(path => {
              if (!cacheKeyMap.has(path)) {
                throw Error("Unknown absolute import path ".concat(path));
              }
              cacheEntry.cacheKeys[path] = cacheKeyMap.get(path);
            });

            // Save the cache entry.
            this._cache.set(cacheKey, cacheEntry);
            this._writeCacheAsync(cacheKey, cacheEntry);
          }
          return cacheEntry.compileResult;
        };
        if (this.compileOneFileLater && inputFile.supportsLazyCompilation) {
          if (!this.isRoot(inputFile)) {
            // If this inputFile is definitely not a root, then it must be
            // lazy, and this is our last chance to mark it as such, so that
            // the rest of the compiler plugin system can avoid worrying
            // about the MultiFileCachingCompiler-specific concept of a
            // "root." If this.isRoot(inputFile) returns true instead, that
            // classification may not be trustworthy, since returning true
            // used to be the only way to get the file to be compiled, so
            // that it could be imported later by a JS module. Now that
            // files can be compiled on-demand, it's safe to pass all files
            // that might be roots to this.compileOneFileLater.
            inputFile.getFileOptions().lazy = true;
          }
          await this.compileOneFileLater(inputFile, getResult);
        } else if (this.isRoot(inputFile)) {
          const result = await getResult();
          if (result) {
            await this.addCompileResult(inputFile, result);
          }
        }
      }
      if (this._cacheDebugEnabled) {
        this._afterLinkCallbacks.push(() => {
          cacheMisses.sort();
          this._cacheDebug("Ran (#".concat(++this._callCount, ") on: ").concat(JSON.stringify(cacheMisses), " ").concat(JSON.stringify(Object.keys(arches).sort())));
        });
      }
    }

    // Returns a hash that incorporates both this.getCacheKey(inputFile) and
    // this.getAbsoluteImportPath(inputFile), since the file path might be
    // relevant to the compiled output when using MultiFileCachingCompiler.
    _getCacheKeyWithPath(inputFile) {
      return this._deepHash([this.getAbsoluteImportPath(inputFile), this.getCacheKey(inputFile)]);
    }
    _cacheEntryValid(cacheEntry, cacheKeyMap) {
      return Object.keys(cacheEntry.cacheKeys).every(path => cacheEntry.cacheKeys[path] === cacheKeyMap.get(path));
    }

    // The format of a cache file on disk is the JSON-stringified cacheKeys
    // object, a newline, followed by the CompileResult as returned from
    // this.stringifyCompileResult.
    _cacheFilename(cacheKey) {
      return path.join(this._diskCache, cacheKey + ".cache");
    }

    // Loads a {compileResult, cacheKeys} cache entry from disk. Returns the whole
    // cache entry and loads it into the in-memory cache too.
    _readCache(cacheKey) {
      if (!this._diskCache) {
        return null;
      }
      const cacheFilename = this._cacheFilename(cacheKey);
      const raw = this._readFileOrNull(cacheFilename);
      if (!raw) {
        return null;
      }

      // Split on newline.
      const newlineIndex = raw.indexOf('\n');
      if (newlineIndex === -1) {
        return null;
      }
      const cacheKeysString = raw.substring(0, newlineIndex);
      const compileResultString = raw.substring(newlineIndex + 1);
      const cacheKeys = this._parseJSONOrNull(cacheKeysString);
      if (!cacheKeys) {
        return null;
      }
      const compileResult = this.parseCompileResult(compileResultString);
      if (!compileResult) {
        return null;
      }
      const cacheEntry = {
        compileResult,
        cacheKeys
      };
      this._cache.set(cacheKey, cacheEntry);
      return cacheEntry;
    }
    _writeCacheAsync(cacheKey, cacheEntry) {
      if (!this._diskCache) {
        return null;
      }
      const cacheFilename = this._cacheFilename(cacheKey);
      const cacheContents = JSON.stringify(cacheEntry.cacheKeys) + '\n' + this.stringifyCompileResult(cacheEntry.compileResult);
      this._writeFile(cacheFilename, cacheContents);
    }
  };
}.call(this, module);
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

}}}}},{
  "extensions": [
    ".js",
    ".json"
  ]
});


/* Exports */
return {
  export: function () { return {
      CachingCompiler: CachingCompiler,
      MultiFileCachingCompiler: MultiFileCachingCompiler
    };},
  require: require,
  eagerModulePaths: [
    "/node_modules/meteor/caching-compiler/caching-compiler.js",
    "/node_modules/meteor/caching-compiler/multi-file-caching-compiler.js"
  ]
}});




//# sourceURL=meteor://💻app/packages/caching-compiler.js
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
